import { $ } from "bun";
import { join, basename } from "path";
import { mkdtemp, rm } from "fs/promises";
import { tmpdir } from "os";

const REMOTE_HOST = "nvps.jeantinland.com";
const REMOTE_USER = "jean";
const REMOTE_BASE_PATH = "packages";

interface PackageJson {
  name: string;
  version: string;
  files?: string[];
  scripts?: {
    prepublishOnly?: string;
    [key: string]: string | undefined;
  };
}

interface PublishOptions {
  dryRun?: boolean;
}

export async function publish(options: PublishOptions = {}): Promise<void> {
  const { dryRun = false } = options;
  const cwd = process.cwd();

  // 1. Read and validate package.json
  console.log("📦 Reading package.json...");
  const packageJsonPath = join(cwd, "package.json");
  const packageJsonFile = Bun.file(packageJsonPath);

  if (!(await packageJsonFile.exists())) {
    throw new Error("package.json not found in current directory");
  }

  const packageJson: PackageJson = await packageJsonFile.json();

  if (!packageJson.name) {
    throw new Error("package.json must have a 'name' field");
  }

  if (!packageJson.version) {
    throw new Error("package.json must have a 'version' field");
  }

  if (!packageJson.files || packageJson.files.length === 0) {
    throw new Error(
      "package.json must have a 'files' field with at least one entry"
    );
  }

  const { name, version, files } = packageJson;
  const packageName = name.startsWith("@")
    ? name.slice(1).replace("/", "-")
    : name;

  console.log(`📋 Package: ${name}@${version}`);
  console.log(`📁 Files to include: ${files.join(", ")}`);

  // 2. Check if version already exists on remote
  const remoteDir = `${REMOTE_BASE_PATH}/${packageName}`;
  const remotePath = `${remoteDir}/${version}.tar.gz`;
  console.log(`\n🔍 Checking if ${remotePath} already exists on remote...`);

  const checkResult =
    await $`ssh ${REMOTE_USER}@${REMOTE_HOST} "test -f ${remotePath} && echo 'exists' || echo 'not-exists'"`.quiet();
  const remoteExists = checkResult.stdout.toString().trim() === "exists";

  if (remoteExists) {
    throw new Error(
      `Version ${version} already exists on the server at ${remotePath}.\n` +
        `Please bump the version in package.json before publishing.`
    );
  }

  console.log("✅ Version does not exist on remote, proceeding...");

  // 3. Run prepublishOnly script if it exists
  if (packageJson.scripts?.prepublishOnly) {
    console.log("\n⚙️  Running prepublishOnly script...");
    const prepublishResult = await $`bun run prepublishOnly`.cwd(cwd);

    if (prepublishResult.exitCode !== 0) {
      throw new Error(
        `prepublishOnly script failed:\n${prepublishResult.stderr.toString()}`
      );
    }

    console.log("✅ prepublishOnly completed successfully");
  }

  // 4. Create temporary directory and archive
  const tempDir = await mkdtemp(join(tmpdir(), "jpub-"));
  const archiveName = `${version}.tar.gz`;
  const archivePath = join(tempDir, archiveName);

  try {
    // Always include package.json in the archive
    const filesToArchive = [...new Set(["package.json", ...files])];

    console.log(`\n📦 Creating archive: ${archiveName}`);

    // Create tar.gz archive with all files in a package/ directory
    const tarResult = await $`tar -czvf ${archivePath} -s '|^|package/|' ${filesToArchive}`.cwd(
      cwd
    );

    if (tarResult.exitCode !== 0) {
      throw new Error(
        `Failed to create archive: ${tarResult.stderr.toString()}`
      );
    }

    console.log("✅ Archive created successfully");

    if (dryRun) {
      console.log("\n🏃 Dry run mode - skipping upload");
      console.log(`   Would upload: ${archivePath}`);
      console.log(`   To: ${REMOTE_USER}@${REMOTE_HOST}:${remotePath}`);
      return;
    }

    // 5. Create remote directory and upload
    console.log(`\n📤 Creating remote directory and uploading...`);

    // Create remote directory
    await $`ssh ${REMOTE_USER}@${REMOTE_HOST} "mkdir -p ${remoteDir}"`.quiet();

    // Upload using rsync (better for reliability and progress)
    const rsyncResult =
      await $`rsync -avz --progress ${archivePath} ${REMOTE_USER}@${REMOTE_HOST}:${remotePath}`;

    if (rsyncResult.exitCode !== 0) {
      throw new Error(`Failed to upload: ${rsyncResult.stderr.toString()}`);
    }

    // Extract on remote and clean up archive

    console.log(`\n✨ Successfully published ${name}@${version}`);
    console.log(`   📍 Location: ${REMOTE_USER}@${REMOTE_HOST}:${remotePath}`);
    console.log(
      `   🔗 URL: https://packages.jeantinland.com/${remotePath.replace(
        "packages/",
        ""
      )}`
    );
  } finally {
    // Clean up temporary directory
    await rm(tempDir, { recursive: true, force: true });
  }
}
