import { $ } from "bun";
import { join } from "path";
import { homedir } from "os";

export async function installCompletions(): Promise<void> {
  console.log("🔧 Installing shell completions...\n");

  const completionsDir = new URL("../completions", import.meta.url).pathname;
  const shell = process.env.SHELL || "";
  const home = homedir();

  try {
    if (shell.includes("fish")) {
      await installFishCompletions(completionsDir, home);
    } else if (shell.includes("zsh")) {
      await installZshCompletions(completionsDir, home);
    } else if (shell.includes("bash")) {
      await installBashCompletions(completionsDir, home);
    } else {
      console.log("⚠️  Could not detect shell. Please install manually:");
      printManualInstructions(completionsDir);
      return;
    }
  } catch (error) {
    console.error(`\x1b[31mError:\x1b[0m ${error instanceof Error ? error.message : error}`);
    console.log("\nYou can install completions manually:");
    printManualInstructions(completionsDir);
  }
}

async function installFishCompletions(completionsDir: string, home: string): Promise<void> {
  const configDir = join(home, ".config", "fish", "completions");
  const targetFile = join(configDir, "jpub.fish");

  await $`mkdir -p ${configDir}`.quiet();
  await $`cp ${join(completionsDir, "jpub.fish")} ${targetFile}`.quiet();

  console.log("✅ Fish completions installed!");
  console.log(`   📁 ${targetFile}`);
  console.log("\n💡 Restart your shell or run: source ${targetFile}");
}

async function installZshCompletions(completionsDir: string, home: string): Promise<void> {
  const zshrcPath = join(home, ".zshrc");
  const targetDir = join(home, ".zsh", "completions");
  const targetFile = join(targetDir, "_jpub");

  await $`mkdir -p ${targetDir}`.quiet();
  await $`cp ${join(completionsDir, "jpub.zsh")} ${targetFile}`.quiet();

  console.log("✅ Zsh completions installed!");
  console.log(`   📁 ${targetFile}`);
  console.log("\n💡 Add this to your ~/.zshrc if not already present:");
  console.log(`   fpath=(~/.zsh/completions $fpath)`);
  console.log(`   autoload -Uz compinit && compinit`);
  console.log("\n   Then restart your shell or run: source ~/.zshrc");
}

async function installBashCompletions(completionsDir: string, home: string): Promise<void> {
  const bashrcPath = join(home, ".bashrc");
  const targetDir = join(home, ".bash_completions");
  const targetFile = join(targetDir, "jpub.bash");

  await $`mkdir -p ${targetDir}`.quiet();
  await $`cp ${join(completionsDir, "jpub.bash")} ${targetFile}`.quiet();

  console.log("✅ Bash completions installed!");
  console.log(`   📁 ${targetFile}`);
  console.log("\n💡 Add this to your ~/.bashrc if not already present:");
  console.log(`   source ~/.bash_completions/jpub.bash`);
  console.log("\n   Then restart your shell or run: source ~/.bashrc");
}

function printManualInstructions(completionsDir: string): void {
  console.log("\n📋 Manual installation:");
  console.log(`\n  Bash: Add to ~/.bashrc:`);
  console.log(`    source ${join(completionsDir, "jpub.bash")}`);
  console.log(`\n  Zsh: Copy ${join(completionsDir, "jpub.zsh")} to your fpath as _jpub`);
  console.log(`\n  Fish: Copy ${join(completionsDir, "jpub.fish")} to ~/.config/fish/completions/`);
}
