import { describe, it, expect, vi, beforeEach } from "vitest";
import sitemap from "../sitemap";
import * as API from "../api";
import { getConfig } from "../config";
vi.mock("../api", () => ({
    getExportPathMap: vi.fn(),
    getSeo: vi.fn(),
}));
vi.mock("../config", () => ({
    getConfig: vi.fn(() => ({
        edenApi: "https://api.test.com",
        site: "test-site",
        langs: ["en"],
        langPrefix: false,
        domains: {},
        siteName: "Test",
        preProd: false,
    })),
}));
describe("Sitemap Service", () => {
    beforeEach(() => {
        vi.clearAllMocks();
    });
    describe("sitemap generation", () => {
        it("should return empty array when preProd is true", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: true,
            });
            const result = await sitemap();
            expect(result).toEqual([]);
            expect(API.getExportPathMap).not.toHaveBeenCalled();
        });
        it("should generate sitemap with single language", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
                "/en/about": "about",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Test",
                description: "Test",
                priority: 0.8,
                frequency: "Weekly",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(Array.isArray(result)).toBe(true);
            expect(result.length).toBeGreaterThan(0);
            expect(result[0]).toHaveProperty("url");
            expect(result[0]).toHaveProperty("lastModified");
            expect(result[0]).toHaveProperty("changeFrequency");
            expect(result[0]).toHaveProperty("priority");
        });
        it("should exclude pages with inSitemap false", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
                "/en/private": "private",
            });
            vi.mocked(API.getSeo).mockImplementation(async (host, site, lang, path) => {
                if (path === "/private") {
                    return {
                        title: "Private",
                        inSitemap: false,
                        last_updated: "2024-01-01",
                    };
                }
                return {
                    title: "Home",
                    inSitemap: true,
                    last_updated: "2024-01-01",
                };
            });
            const result = await sitemap();
            expect(result.every((item) => !item.url.includes("private"))).toBe(true);
        });
        it("should handle multiple languages", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en", "fr"],
                langPrefix: false,
                domains: { en: "example.com", fr: "example.fr" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
                "/fr/": "index",
                "/en/about": "about",
                "/fr/about": "about",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Test",
                priority: 0.8,
                frequency: "Weekly",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result.length).toBe(4);
        });
        it("should exclude languages in preProd array", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en", "fr"],
                langPrefix: false,
                domains: { en: "example.com", fr: "example.fr" },
                siteName: "Test",
                preProd: ["en"],
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/fr/": "index",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Test",
                priority: 0.8,
                frequency: "Always",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            await sitemap();
            expect(API.getExportPathMap).toHaveBeenCalledWith("https://api.test.com", "test-site", "fr", false);
        });
        it("should use default priority for root path", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Home",
                priority: undefined,
                frequency: "Weekly",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result[0].priority).toBe(1);
        });
        it("should calculate priority based on depth", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/about/team/members/": "members",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Members",
                priority: undefined,
                frequency: "Monthly",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result[0].priority).toBe(0.7);
        });
        it("should use provided priority over calculated", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/about": "about",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "About",
                priority: 0.9,
                frequency: "Yearly",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result[0].priority).toBe(0.9);
        });
        it("should use Always as default frequency", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Home",
                priority: 0.8,
                frequency: undefined,
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result[0].changeFrequency).toBe("Always");
        });
        it("should construct URL with correct domain and path", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en", "fr"],
                langPrefix: false,
                domains: { en: "example.com", fr: "example.fr" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/about": "about",
                "/fr/about": "about",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "About",
                priority: 0.8,
                frequency: "Weekly",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result[0].url).toBe("https://example.com/about");
            expect(result[1].url).toBe("https://example.fr/about");
        });
        it("should handle complex path transformations", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/blog/posts/my-post": "post",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "My Post",
                priority: 0.7,
                frequency: "Never",
                inSitemap: true,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result[0].url).toBe("https://example.com/blog/posts/my-post");
        });
        it("should convert lastUpdated to Date object", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Home",
                priority: 0.8,
                frequency: "Weekly",
                inSitemap: true,
                last_updated: "2024-12-22",
            });
            const result = await sitemap();
            expect(result[0].lastModified).toBeInstanceOf(Date);
        });
        it("should use inSitemap default as true", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
            });
            vi.mocked(API.getSeo).mockResolvedValue({
                title: "Home",
                priority: 0.8,
                frequency: "Weekly",
                inSitemap: undefined,
                last_updated: "2024-01-01",
            });
            const result = await sitemap();
            expect(result.length).toBe(1);
        });
        it("should handle empty export path map", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test-site",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({});
            const result = await sitemap();
            expect(result).toEqual([]);
        });
    });
});
