import { describe, it, expect, vi, beforeEach } from "vitest";
import revalidate from "../revalidate";
import * as API from "../api";
import { getConfig } from "../config";
vi.mock("../api", () => ({
    checkToken: vi.fn(),
    getExportPathMap: vi.fn(),
}));
vi.mock("../config", () => ({
    getConfig: vi.fn(() => ({
        edenApi: "https://api.test.com",
        site: "test-site",
        langs: ["en"],
        langPrefix: false,
        domains: {},
        siteName: "Test",
        preProd: false,
    })),
}));
describe("Revalidate Service", () => {
    let mockRes;
    beforeEach(() => {
        vi.clearAllMocks();
        mockRes = {
            status: vi.fn().mockReturnThis(),
            send: vi.fn().mockReturnThis(),
            revalidate: vi.fn(),
        };
        vi.mocked(getConfig).mockReturnValue({
            edenApi: "https://api.test.com",
            site: "test-site",
            langs: ["en"],
            langPrefix: false,
            domains: {},
            siteName: "Test",
            preProd: false,
        });
    });
    describe("revalidate handler", () => {
        it("should return 403 if no token provided", async () => {
            const req = { query: { path: "/test" } };
            await revalidate(req, mockRes);
            expect(mockRes.status).toHaveBeenCalledWith(403);
            expect(mockRes.send).toHaveBeenCalledWith({
                error: "no token provided",
            });
        });
        it("should return 403 if token is invalid", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-invalid",
            });
            const req = {
                query: { token: "invalid-token", path: "/test" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.status).toHaveBeenCalledWith(403);
            expect(mockRes.send).toHaveBeenCalledWith({
                error: "invalid token",
            });
        });
        it("should verify token with checkToken", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            const req = {
                query: { token: "valid-token", path: "/test" },
            };
            await revalidate(req, mockRes);
            expect(API.checkToken).toHaveBeenCalledWith("https://api.test.com", "valid-token");
        });
        it("should revalidate single path", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            const req = {
                query: { token: "valid-token", path: "/test" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.revalidate).toHaveBeenCalledWith("/test");
            expect(mockRes.send).toHaveBeenCalledWith({ status: "revalidated" });
        });
        it("should handle path as array", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            const req = {
                query: {
                    token: "valid-token",
                    path: ["about", "team"],
                },
            };
            await revalidate(req, mockRes);
            expect(mockRes.revalidate).toHaveBeenCalledWith("/about/team");
        });
        it("should revalidate all paths when all flag is set", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
                "/en/about": "about",
                "/en/contact": "contact",
            });
            const req = {
                query: { token: "valid-token", all: "true", lang: "en" },
            };
            await revalidate(req, mockRes);
            expect(API.getExportPathMap).toHaveBeenCalledWith("https://api.test.com", "test-site", "en");
            expect(mockRes.revalidate).toHaveBeenCalledTimes(3);
            expect(mockRes.send).toHaveBeenCalledWith({
                status: "revalidated all",
            });
        });
        it("should return 400 when all flag is set but no lang provided", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            const req = {
                query: { token: "valid-token", all: "true" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.status).toHaveBeenCalledWith(400);
            expect(mockRes.send).toHaveBeenCalledWith({
                error: "no lang provided",
            });
        });
        it("should return 400 when neither path nor all is provided", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            const req = {
                query: { token: "valid-token" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.status).toHaveBeenCalledWith(400);
            expect(mockRes.send).toHaveBeenCalledWith({
                error: "no path provided",
            });
        });
        it("should handle lang as string", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
            });
            const req = {
                query: { token: "valid-token", all: "true", lang: "en" },
            };
            await revalidate(req, mockRes);
            expect(API.getExportPathMap).toHaveBeenCalledWith("https://api.test.com", "test-site", "en");
        });
        it("should handle token as array and join it", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            const req = {
                query: {
                    token: ["part1", "part2"],
                    path: "/test",
                },
            };
            await revalidate(req, mockRes);
            expect(API.checkToken).toHaveBeenCalledWith("https://api.test.com", "part1part2");
        });
        it("should return 500 on unexpected error", async () => {
            const consoleErrorSpy = vi
                .spyOn(console, "error")
                .mockImplementation(() => { });
            vi.mocked(API.checkToken).mockRejectedValue(new Error("Network error"));
            const req = {
                query: { token: "valid-token", path: "/test" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.status).toHaveBeenCalledWith(500);
            expect(mockRes.send).toHaveBeenCalledWith({
                error: "server error",
            });
            expect(consoleErrorSpy).toHaveBeenCalled();
            consoleErrorSpy.mockRestore();
        });
        it("should process multiple paths from export path map", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            const exportMap = {
                "/en/": "index",
                "/en/about": "about",
                "/en/contact": "contact",
                "/en/blog": "blog",
                "/en/blog/post-1": "post-1",
            };
            vi.mocked(API.getExportPathMap).mockResolvedValue(exportMap);
            const req = {
                query: { token: "valid-token", all: "true", lang: "en" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.revalidate).toHaveBeenCalledTimes(5);
        });
        it("should handle lang as array in query", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({});
            const req = {
                query: { token: "valid-token", all: "true", lang: ["en"] },
            };
            await revalidate(req, mockRes);
            expect(API.getExportPathMap).toHaveBeenCalled();
        });
        it("should remove language prefix from path", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/about": "about",
                "/en/contact": "contact",
            });
            const req = {
                query: { token: "valid-token", all: "true", lang: "en" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.revalidate).toHaveBeenCalledWith("/about/");
            expect(mockRes.revalidate).toHaveBeenCalledWith("/contact/");
        });
        it("should handle paths with trailing slash", async () => {
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(API.getExportPathMap).mockResolvedValue({
                "/en/": "index",
            });
            const req = {
                query: { token: "valid-token", all: "true", lang: "en" },
            };
            await revalidate(req, mockRes);
            expect(mockRes.revalidate).toHaveBeenCalledWith("/");
        });
    });
});
