import { describe, it, expect, vi, beforeEach } from "vitest";
import * as Fetcher from "../fetcher";
import * as HTTP from "../../utils/http";
vi.mock("../../utils/http", () => ({
    GET: vi.fn(),
    getText: vi.fn(),
}));
describe("Fetcher Service", () => {
    beforeEach(() => {
        vi.clearAllMocks();
    });
    describe("fetchZone", () => {
        it("should fetch zone with default skipReverse", async () => {
            const mockZone = "<div>Zone content</div>";
            vi.mocked(HTTP.getText).mockResolvedValue(mockZone);
            const result = await Fetcher.fetchZone("https://api.test.com", "test-site", "en", "/test", "header");
            expect(HTTP.getText).toHaveBeenCalledWith("https://api.test.com", "/zone", {
                site: "test-site",
                lang: "en",
                path: "/test",
                id: "header",
                skipReverse: "false",
            });
            expect(result).toBe(mockZone);
        });
        it("should fetch zone with skipReverse true", async () => {
            const mockZone = "<div>Zone content</div>";
            vi.mocked(HTTP.getText).mockResolvedValue(mockZone);
            await Fetcher.fetchZone("https://api.test.com", "test-site", "en", "/test", "header", true);
            expect(HTTP.getText).toHaveBeenCalledWith("https://api.test.com", "/zone", {
                site: "test-site",
                lang: "en",
                path: "/test",
                id: "header",
                skipReverse: "true",
            });
        });
        it("should fetch zone with skipReverse false explicitly", async () => {
            vi.mocked(HTTP.getText).mockResolvedValue("");
            await Fetcher.fetchZone("https://api.test.com", "test-site", "en", "/test", "sidebar", false);
            expect(HTTP.getText).toHaveBeenCalledWith("https://api.test.com", "/zone", {
                site: "test-site",
                lang: "en",
                path: "/test",
                id: "sidebar",
                skipReverse: "false",
            });
        });
        it("should handle different zone IDs", async () => {
            vi.mocked(HTTP.getText).mockResolvedValue("<div>Content</div>");
            await Fetcher.fetchZone("https://api.test.com", "test-site", "en", "/", "footer");
            expect(HTTP.getText).toHaveBeenCalledWith("https://api.test.com", "/zone", expect.objectContaining({ id: "footer" }));
        });
    });
    describe("fetchZones", () => {
        it("should fetch all zones with default skipReverse", async () => {
            const mockZones = {
                header: "<div>Header</div>",
                content: "<div>Content</div>",
            };
            vi.mocked(HTTP.GET).mockResolvedValue(mockZones);
            const result = await Fetcher.fetchZones("https://api.test.com", "test-site", "en", "/test");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/zones", {
                site: "test-site",
                lang: "en",
                path: "/test",
                skipReverse: "false",
            });
            expect(result).toEqual(mockZones);
        });
        it("should fetch zones with skipReverse true", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue({});
            await Fetcher.fetchZones("https://api.test.com", "test-site", "en", "/test", true);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/zones", {
                site: "test-site",
                lang: "en",
                path: "/test",
                skipReverse: "true",
            });
        });
        it("should handle different languages", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue({});
            await Fetcher.fetchZones("https://api.test.com", "test-site", "fr", "/about");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/zones", expect.objectContaining({ lang: "fr", path: "/about" }));
        });
        it("should return multiple zones", async () => {
            const mockZones = {
                header: "<div>Header</div>",
                content: "<div>Content</div>",
                footer: "<div>Footer</div>",
                sidebar: "<div>Sidebar</div>",
            };
            vi.mocked(HTTP.GET).mockResolvedValue(mockZones);
            const result = await Fetcher.fetchZones("https://api.test.com", "test-site", "en", "/page");
            expect(Object.keys(result)).toHaveLength(4);
        });
    });
    describe("fetchMenus", () => {
        it("should fetch menus with default parameters", async () => {
            const mockMenus = [{ path: "/", title: "Home", next: null, prev: null }];
            vi.mocked(HTTP.GET).mockResolvedValue(mockMenus);
            const result = await Fetcher.fetchMenus("https://api.test.com", "test-site", "en", "/");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/menus", {
                site: "test-site",
                lang: "en",
                from: "/",
                depth: "1",
                root: "false",
            });
            expect(result).toEqual(mockMenus);
        });
        it("should fetch menus with custom depth", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue([]);
            await Fetcher.fetchMenus("https://api.test.com", "test-site", "en", "/", 3);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/menus", expect.objectContaining({ depth: "3" }));
        });
        it("should fetch menus with root parameter", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue([]);
            await Fetcher.fetchMenus("https://api.test.com", "test-site", "en", "/about", 2, true);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/menus", expect.objectContaining({ root: "true", depth: "2" }));
        });
        it("should handle different languages and paths", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue([]);
            await Fetcher.fetchMenus("https://api.test.com", "test-site", "fr", "/contact");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/menus", expect.objectContaining({ lang: "fr", from: "/contact" }));
        });
        it("should return array of menu items", async () => {
            const mockMenus = [
                { path: "/", title: "Home", next: null, prev: null },
                { path: "/about", title: "About", next: null, prev: null },
                { path: "/contact", title: "Contact", next: null, prev: null },
            ];
            vi.mocked(HTTP.GET).mockResolvedValue(mockMenus);
            const result = await Fetcher.fetchMenus("https://api.test.com", "test-site", "en", "/");
            expect(Array.isArray(result)).toBe(true);
            expect(result).toHaveLength(3);
        });
        it("should handle deep depth values", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue([]);
            await Fetcher.fetchMenus("https://api.test.com", "test-site", "en", "/", 10);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/menus", expect.objectContaining({ depth: "10" }));
        });
    });
    describe("fetchLinks", () => {
        it("should return empty array when menus array is empty", async () => {
            const result = await Fetcher.fetchLinks("https://api.test.com", "test-site", "en", []);
            expect(result).toEqual([]);
            expect(HTTP.GET).not.toHaveBeenCalled();
        });
        it("should return empty array when menus is undefined", async () => {
            const result = await Fetcher.fetchLinks("https://api.test.com", "test-site", "en", undefined);
            expect(result).toEqual([]);
            expect(HTTP.GET).not.toHaveBeenCalled();
        });
        it("should fetch links with single menu", async () => {
            const mockLinks = [
                { id: "1", title: "Link 1", url: "/link1" },
                { id: "2", title: "Link 2", url: "/link2" },
            ];
            vi.mocked(HTTP.GET).mockResolvedValue(mockLinks);
            const result = await Fetcher.fetchLinks("https://api.test.com", "test-site", "en", ["home"]);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/links", {
                site: "test-site",
                lang: "en",
                menus: "home",
            });
            expect(result).toEqual(mockLinks);
        });
        it("should fetch links with multiple menus", async () => {
            const mockLinks = [];
            vi.mocked(HTTP.GET).mockResolvedValue(mockLinks);
            await Fetcher.fetchLinks("https://api.test.com", "test-site", "en", [
                "home",
                "about",
                "contact",
            ]);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/links", {
                site: "test-site",
                lang: "en",
                menus: "home,about,contact",
            });
        });
        it("should handle different languages", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue([]);
            await Fetcher.fetchLinks("https://api.test.com", "test-site", "fr", [
                "accueil",
            ]);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/links", expect.objectContaining({ lang: "fr" }));
        });
        it("should join multiple menus with comma", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue([]);
            await Fetcher.fetchLinks("https://api.test.com", "test-site", "en", [
                "menu1",
                "menu2",
                "menu3",
                "menu4",
            ]);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/links", expect.objectContaining({ menus: "menu1,menu2,menu3,menu4" }));
        });
    });
});
