import { describe, it, expect, vi, beforeEach } from "vitest";
import draft from "../draft";
import * as API from "../api";
import { getConfig } from "../config";
import { draftMode } from "next/headers";
vi.mock("../api", () => ({
    checkToken: vi.fn(),
}));
vi.mock("../config", () => ({
    getConfig: vi.fn(() => ({
        edenApi: "https://api.test.com",
        site: "test-site",
        langs: ["en"],
        langPrefix: false,
        domains: {},
        siteName: "Test",
        preProd: false,
    })),
}));
vi.mock("next/headers", () => ({
    draftMode: vi.fn(),
}));
describe("Draft Service", () => {
    beforeEach(() => {
        vi.clearAllMocks();
    });
    describe("draft handler", () => {
        it("should return 403 if no token provided", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test",
                langs: ["en"],
                langPrefix: false,
                domains: {},
                siteName: "Test",
                preProd: false,
            });
            const request = new Request("http://localhost:3000/api/draft");
            const response = await draft(request);
            expect(response.status).toBe(403);
            const body = await response.json();
            expect(body.error).toBe("no token provided");
        });
        it("should return 403 if token is invalid", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test",
                langs: ["en"],
                langPrefix: false,
                domains: {},
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-invalid",
            });
            const request = new Request("http://localhost:3000/api/draft?token=invalid-token");
            const response = await draft(request);
            expect(response.status).toBe(403);
            const body = await response.json();
            expect(body.error).toBe("invalid token");
        });
        it("should enable draft mode when status is true", async () => {
            const mockDraft = { enable: vi.fn(), disable: vi.fn(), isEnabled: false };
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test",
                langs: ["en"],
                langPrefix: false,
                domains: {},
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(draftMode).mockResolvedValue(mockDraft);
            const request = new Request("http://localhost:3000/api/draft?token=valid-token&status=true");
            const response = await draft(request);
            expect(response.status).toBe(200);
            expect(mockDraft.enable).toHaveBeenCalled();
            const body = await response.json();
            expect(body.message).toBe("Draft mode enabled");
        });
        it("should disable draft mode when status is false", async () => {
            const mockDraft = { enable: vi.fn(), disable: vi.fn(), isEnabled: true };
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test",
                langs: ["en"],
                langPrefix: false,
                domains: {},
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(draftMode).mockResolvedValue(mockDraft);
            const request = new Request("http://localhost:3000/api/draft?token=valid-token&status=false");
            const response = await draft(request);
            expect(response.status).toBe(200);
            expect(mockDraft.disable).toHaveBeenCalled();
            const body = await response.json();
            expect(body.message).toBe("Draft mode disabled");
        });
        it("should verify token with checkToken", async () => {
            const mockDraft = { enable: vi.fn(), disable: vi.fn(), isEnabled: false };
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test",
                langs: ["en"],
                langPrefix: false,
                domains: {},
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(draftMode).mockResolvedValue(mockDraft);
            const request = new Request("http://localhost:3000/api/draft?token=test-token&status=true");
            await draft(request);
            expect(API.checkToken).toHaveBeenCalledWith("https://api.test.com", "test-token");
        });
        it("should return 500 on unexpected error", async () => {
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test",
                langs: ["en"],
                langPrefix: false,
                domains: {},
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.checkToken).mockRejectedValue(new Error("Network error"));
            const request = new Request("http://localhost:3000/api/draft?token=test-token&status=true");
            const response = await draft(request);
            expect(response.status).toBe(500);
            const body = await response.json();
            expect(body.error).toBe("server error");
        });
        it("should handle status parameter variations", async () => {
            const mockDraft = { enable: vi.fn(), disable: vi.fn(), isEnabled: false };
            vi.mocked(getConfig).mockReturnValue({
                edenApi: "https://api.test.com",
                site: "test",
                langs: ["en"],
                langPrefix: false,
                domains: {},
                siteName: "Test",
                preProd: false,
            });
            vi.mocked(API.checkToken).mockResolvedValue({
                message: "token-valid",
            });
            vi.mocked(draftMode).mockResolvedValue(mockDraft);
            const request = new Request("http://localhost:3000/api/draft?token=test-token&status=false");
            await draft(request);
            expect(mockDraft.disable).toHaveBeenCalled();
        });
    });
});
