import { describe, it, expect } from "vitest";
import * as Breadcrumb from "../breadcrumb";
const createItem = (id, name, title, path, next = null, prev = null) => ({
    id,
    active: false,
    disabled: false,
    name,
    title,
    menu: "main",
    path,
    next,
    prev,
});
describe("Breadcrumb Service", () => {
    describe("reverse", () => {
        it("should return null for null input", () => {
            const result = Breadcrumb.reverse(null);
            expect(result).toBeNull();
        });
        it("should return single item with null next as is", () => {
            const item = createItem(1, "test", "Test", "/test");
            const result = Breadcrumb.reverse(item);
            expect(result).toEqual(item);
        });
    });
    describe("deepClone", () => {
        it("should return null for null input", () => {
            const result = Breadcrumb.deepClone(null);
            expect(result).toBeNull();
        });
        it("should clone a single item", () => {
            const item = createItem(1, "test", "Test", "/test");
            const result = Breadcrumb.deepClone(item);
            expect(result).toEqual(item);
            expect(result).not.toBe(item);
        });
        it("should deep clone a chain of items", () => {
            const item1 = createItem(1, "home", "Home", "/");
            const item2 = createItem(2, "about", "About", "/about", item1);
            const item3 = createItem(3, "team", "Team", "/about/team", item2);
            const result = Breadcrumb.deepClone(item3);
            expect(result).toEqual(item3);
            expect(result).not.toBe(item3);
            expect(result?.next).not.toBe(item2);
            expect(result?.next?.next).not.toBe(item1);
        });
        it("should clone all properties of items", () => {
            const item1 = createItem(1, "home", "Home", "/");
            const item2 = createItem(2, "about", "About", "/about", item1);
            const result = Breadcrumb.deepClone(item2);
            expect(result?.path).toBe("/about");
            expect(result?.title).toBe("About");
            expect(result?.next?.path).toBe("/");
            expect(result?.next?.title).toBe("Home");
        });
        it("should handle long chains", () => {
            let chain = null;
            for (let i = 0; i < 10; i++) {
                chain = createItem(i, `path${i}`, `Path ${i}`, `/path${i}`, chain);
            }
            const result = Breadcrumb.deepClone(chain);
            let current = result;
            let count = 0;
            while (current) {
                count++;
                current = current.next;
            }
            expect(count).toBe(10);
        });
    });
});
