import { describe, it, expect, vi, beforeEach } from "vitest";
import { renderHook, waitFor } from "@testing-library/react";
import useDateHiddenBlocks from "../use-date-hidden-blocks";
import * as Context from "../../components/context";
vi.mock("next/navigation", () => ({
    usePathname: vi.fn(() => "/test-path"),
}));
vi.mock("../../components/context", async () => {
    const actual = await vi.importActual("../../components/context");
    return {
        ...actual,
        useEdenContext: vi.fn(),
    };
});
describe("useDateHiddenBlocks", () => {
    const mockContext = {
        lang: "en",
        path: "/test",
        edenMode: null,
        config: {},
        pageProps: {},
        zones: {},
        isConnected: false,
        edenApi: "",
        site: "",
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    beforeEach(() => {
        vi.clearAllMocks();
        document.body.innerHTML = "";
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
    });
    it("should hide blocks outside date range", async () => {
        const futureDate = new Date(2099, 11, 31).toISOString();
        const currentDate = new Date().toISOString();
        const block = document.createElement("section");
        block.dataset.dates = JSON.stringify([[currentDate, futureDate]]);
        document.body.appendChild(block);
        renderHook(() => useDateHiddenBlocks());
        await waitFor(() => {
            expect(block.style.display).not.toBe("none");
        });
    });
    it("should show blocks within date range", async () => {
        const pastDate = new Date(2020, 0, 1).toISOString();
        const veryPastDate = new Date(2019, 0, 1).toISOString();
        const block = document.createElement("section");
        block.dataset.dates = JSON.stringify([[veryPastDate, pastDate]]);
        document.body.appendChild(block);
        renderHook(() => useDateHiddenBlocks());
        await waitFor(() => {
            expect(block.style.display).toBe("none");
        });
    });
    it("should not filter when edenMode is active", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            edenMode: "editing",
        });
        const block = document.createElement("section");
        block.dataset.dates = JSON.stringify([
            [new Date(2020, 0, 1).toISOString(), new Date(2020, 1, 1).toISOString()],
        ]);
        document.body.appendChild(block);
        renderHook(() => useDateHiddenBlocks());
        expect(block.style.display).not.toBe("none");
    });
    it("should handle multiple date ranges", async () => {
        const futureStart = new Date(2099, 0, 1).toISOString();
        const futureEnd = new Date(2099, 11, 31).toISOString();
        const pastStart = new Date(2020, 0, 1).toISOString();
        const pastEnd = new Date(2020, 1, 1).toISOString();
        const block = document.createElement("section");
        block.dataset.dates = JSON.stringify([
            [pastStart, pastEnd],
            [futureStart, futureEnd],
        ]);
        document.body.appendChild(block);
        renderHook(() => useDateHiddenBlocks());
        await waitFor(() => {
            expect(block.style.display).toBe("none");
        });
    });
    it("should restore original display style when cleaning up", async () => {
        const block = document.createElement("section");
        block.style.display = "flex";
        block.dataset.dates = JSON.stringify([
            [new Date(2020, 0, 1).toISOString(), new Date(2020, 1, 1).toISOString()],
        ]);
        document.body.appendChild(block);
        const { unmount } = renderHook(() => useDateHiddenBlocks());
        await waitFor(() => {
            expect(block.style.display).toBe("none");
        });
        unmount();
        await waitFor(() => {
            expect(block.style.display).toBe("flex");
        });
    });
    it("should handle blocks without dates attribute", () => {
        const block = document.createElement("section");
        document.body.appendChild(block);
        expect(() => {
            renderHook(() => useDateHiddenBlocks());
        }).not.toThrow();
    });
    it("should handle multiple blocks", async () => {
        const currentDate = new Date().toISOString();
        const futureDate = new Date(2099, 11, 31).toISOString();
        const pastDate = new Date(2020, 0, 1).toISOString();
        const visibleBlock = document.createElement("section");
        visibleBlock.dataset.dates = JSON.stringify([[currentDate, futureDate]]);
        document.body.appendChild(visibleBlock);
        const hiddenBlock = document.createElement("section");
        hiddenBlock.dataset.dates = JSON.stringify([[pastDate, pastDate]]);
        document.body.appendChild(hiddenBlock);
        renderHook(() => useDateHiddenBlocks());
        await waitFor(() => {
            expect(visibleBlock.style.display).not.toBe("none");
            expect(hiddenBlock.style.display).toBe("none");
        });
    });
});
