import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, screen, waitFor } from "@testing-library/react";
import userEvent from "@testing-library/user-event";
import Reset from "../reset";
import * as Context from "../../context";
import * as API from "../../../services/api";
vi.mock("../../context", () => ({
    useEdenContext: vi.fn(),
}));
vi.mock("../../../services/api", () => ({
    requestPasswordReset: vi.fn(),
}));
vi.mock("../../logo", () => ({
    default: () => _jsx("div", { "data-testid": "logo", children: "Logo" }),
}));
vi.mock("../../icons/enter-key", () => ({
    default: () => _jsx("div", { "data-testid": "enter-key-icon", children: "Enter" }),
}));
vi.mock("../../icons/loading", () => ({
    default: () => _jsx("div", { "data-testid": "loading-icon", children: "Loading" }),
}));
describe("Reset", () => {
    const mockContext = {
        lang: "fr",
        edenApi: "https://api.test.com",
        site: "test-site",
        config: {
            langPrefix: false,
            edenApi: "",
            site: "",
            lang: "fr",
            langs: ["fr"],
            domains: {},
            siteName: "",
            preProd: false,
            layout: "",
            layouts: [],
        },
        pageProps: {},
        zones: {},
        isConnected: false,
        path: "",
        edenMode: null,
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    const mockOnBack = vi.fn();
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
    });
    it("should render reset form with French translations", () => {
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        expect(screen.getByLabelText(/E-mail/i)).toBeInTheDocument();
        expect(screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        })).toBeInTheDocument();
        expect(screen.getByRole("button", { name: /Se connecter/i })).toBeInTheDocument();
    });
    it("should render reset form with English translations", () => {
        render(_jsx(Reset, { lang: "en", onBack: mockOnBack }));
        expect(screen.getByLabelText(/Email/i)).toBeInTheDocument();
        expect(screen.getByRole("button", { name: /Send reset email/i })).toBeInTheDocument();
        expect(screen.getByRole("button", { name: /Log in/i })).toBeInTheDocument();
    });
    it("should render logo", () => {
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        expect(screen.getByTestId("logo")).toBeInTheDocument();
    });
    it("should update email input", async () => {
        const user = userEvent.setup();
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const emailInput = screen.getByLabelText(/E-mail/i);
        await user.type(emailInput, "test@example.com");
        expect(emailInput.value).toBe("test@example.com");
    });
    it("should have autofocus on email input", () => {
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const emailInput = screen.getByLabelText(/E-mail/i);
        expect(emailInput).toHaveFocus();
    });
    it("should have correct autocomplete attribute", () => {
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const emailInput = screen.getByLabelText(/E-mail/i);
        expect(emailInput).toHaveAttribute("autocomplete", "email");
    });
    it("should submit reset request successfully", async () => {
        const user = userEvent.setup();
        vi.mocked(API.requestPasswordReset).mockResolvedValue({
            error: null,
        });
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        const submitButton = screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(API.requestPasswordReset).toHaveBeenCalledWith("https://api.test.com", "test-site", "test@example.com", "fr");
        });
        expect(screen.getByText(/Un e-mail de réinitialisation a été envoyé à/i)).toBeInTheDocument();
    });
    it("should clear email after successful submission", async () => {
        const user = userEvent.setup();
        vi.mocked(API.requestPasswordReset).mockResolvedValue({
            error: null,
        });
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const emailInput = screen.getByLabelText(/E-mail/i);
        await user.type(emailInput, "test@example.com");
        expect(emailInput.value).toBe("test@example.com");
        const submitButton = screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(emailInput.value).toBe("");
        });
    });
    it("should display error when user not found", async () => {
        const user = userEvent.setup();
        vi.mocked(API.requestPasswordReset).mockResolvedValue({
            error: { message: "User not found" },
        });
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        await user.type(screen.getByLabelText(/E-mail/i), "notfound@example.com");
        await user.click(screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        }));
        await waitFor(() => {
            expect(screen.getByText(/E-mail ou mot de passe incorrect/i)).toBeInTheDocument();
        });
    });
    it("should display generic error message", async () => {
        const user = userEvent.setup();
        vi.mocked(API.requestPasswordReset).mockResolvedValue({
            error: { message: "Network error" },
        });
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.click(screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        }));
        await waitFor(() => {
            expect(screen.getByText("Network error")).toBeInTheDocument();
        });
    });
    it("should handle exception during reset request", async () => {
        const user = userEvent.setup();
        vi.mocked(API.requestPasswordReset).mockRejectedValue(new Error("Network failure"));
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.click(screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        }));
        await waitFor(() => {
            expect(screen.getByText("An error occurred. Please try again.")).toBeInTheDocument();
        });
    });
    it("should show loading state during submission", async () => {
        const user = userEvent.setup();
        vi.mocked(API.requestPasswordReset).mockImplementation(() => new Promise((resolve) => setTimeout(() => resolve({ error: null }), 100)));
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        const submitButton = screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        });
        await user.click(submitButton);
        expect(submitButton).toBeDisabled();
        expect(screen.getByTestId("loading-icon")).toBeInTheDocument();
    });
    it("should call onBack when back button is clicked", async () => {
        const user = userEvent.setup();
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const backButton = screen.getByRole("button", { name: /Se connecter/i });
        await user.click(backButton);
        expect(mockOnBack).toHaveBeenCalledTimes(1);
    });
    it("should not render back button when onBack is not provided", () => {
        render(_jsx(Reset, { lang: "fr" }));
        expect(screen.queryByRole("button", { name: /Se connecter/i })).not.toBeInTheDocument();
    });
    it("should require email input", () => {
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const emailInput = screen.getByLabelText(/E-mail/i);
        expect(emailInput).toHaveAttribute("required");
    });
    it("should have email input type", () => {
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const emailInput = screen.getByLabelText(/E-mail/i);
        expect(emailInput).toHaveAttribute("type", "email");
    });
    it("should display success message with email variable replacement", async () => {
        const user = userEvent.setup();
        vi.mocked(API.requestPasswordReset).mockResolvedValue({
            error: null,
        });
        render(_jsx(Reset, { lang: "fr", onBack: mockOnBack }));
        const testEmail = "user@example.com";
        await user.type(screen.getByLabelText(/E-mail/i), testEmail);
        const submitButton = screen.getByRole("button", {
            name: /Envoyer l'e-mail de réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText(new RegExp(`Un e-mail de réinitialisation a été envoyé à ${testEmail}`))).toBeInTheDocument();
        });
    });
});
