import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach, afterEach } from "vitest";
import { render, screen } from "@testing-library/react";
import "@testing-library/jest-dom";
vi.mock("next/navigation", () => ({
    notFound: vi.fn(() => {
        throw new Error("notFound");
    }),
    redirect: vi.fn((url) => {
        throw new Error(`redirect:${url}`);
    }),
}));
vi.mock("../../services/api");
vi.mock("../../services/breadcrumb");
vi.mock("../../services/fetcher");
vi.mock("../../utils/layouts");
vi.mock("../../services/config", () => ({
    getConfig: vi.fn(() => ({
        edenApi: "https://api.test.com",
        site: "test-site",
        langs: ["en", "fr"],
        domains: { en: "example.com", fr: "exemple.fr" },
        langPrefix: false,
        siteName: "Test Site",
        preProd: false,
    })),
}));
vi.mock("../context", () => ({
    default: ({ children, value }) => (_jsx("div", { "data-testid": "context-provider", "data-context": JSON.stringify(value), children: children })),
}));
vi.mock("../head", () => ({
    default: () => _jsx("div", { "data-testid": "eden-head", children: "Head" }),
}));
vi.mock("../loader", () => ({
    default: ({ config }) => (_jsx("div", { "data-testid": "eden-loader", "data-config": JSON.stringify(config), children: "Loader" })),
}));
import EdenLayout, { generateStaticParams, dynamicParams } from "../layout";
import * as API from "../../services/api";
import * as Breadcrumb from "../../services/breadcrumb";
import { fetchLinks } from "../../services/fetcher";
import { getConfig } from "../../services/config";
import * as Layouts from "../../utils/layouts";
describe("EdenLayout", () => {
    const mockConfig = {
        edenApi: "https://api.test.com",
        site: "test-site",
        langs: ["en", "fr"],
        domains: { en: "example.com", fr: "exemple.fr" },
        langPrefix: false,
        siteName: "Test Site",
        preProd: false,
    };
    const mockMenuPath = {
        id: 1,
        active: true,
        disabled: false,
        name: "home",
        title: "Home",
        menu: "main",
        path: "/",
        prev: null,
        next: null,
        metadata: { key: "value" },
    };
    const mockBreadcrumb = {
        id: 1,
        active: true,
        disabled: false,
        name: "home",
        title: "Home",
        menu: "main",
        path: "/",
        prev: null,
        next: null,
    };
    const mockPage = {
        seo: {
            title: "Test Page",
            description: "Test Description",
        },
        zones: {
            main: "zone-content",
        },
        breadcrumb: mockBreadcrumb,
    };
    const mockLinks = [
        {
            page: "default-layout",
            title: "Home",
            path: "/",
        },
    ];
    const mockLangSwitches = {
        fr: { path: "/fr/", page: "default-layout", disabled: false },
    };
    const mockImports = {
        "default-layout": Promise.resolve({
            default: () => _jsx("div", { children: "Layout Component" }),
            getProps: async (context) => ({ customProp: "test-value" }),
        }),
    };
    const mockParams = {
        locale: "en",
        path: [""],
    };
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(getConfig).mockReturnValue(mockConfig);
        vi.mocked(fetchLinks).mockResolvedValue(mockLinks);
        vi.mocked(API.getPage).mockResolvedValue(mockPage);
        vi.mocked(Breadcrumb.deepClone).mockReturnValue(mockBreadcrumb);
        vi.mocked(Breadcrumb.reverse).mockReturnValue(mockMenuPath);
        vi.mocked(API.getLangSwitches).mockResolvedValue(mockLangSwitches);
        vi.mocked(Layouts.format).mockReturnValue([
            { name: "Default Layout", view: "default-layout" },
        ]);
    });
    afterEach(() => {
        vi.restoreAllMocks();
    });
    describe("Basic Rendering", () => {
        it("should render successfully with valid props", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { "data-testid": "children", children: "Test Children" }),
            });
            expect(result).toBeDefined();
        });
        it("should render with className prop", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                className: "custom-class",
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
        });
        it("should render successfully and include context provider", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { "data-testid": "test-child", children: "Test Child" }),
            });
            render(result);
            expect(screen.getByTestId("context-provider")).toBeInTheDocument();
        });
    });
    describe("Locale Handling", () => {
        it("should use default locale from langs array when not provided", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getPage).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "en", "/test");
        });
        it("should use provided locale", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "fr", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getPage).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "fr", "/test");
        });
        it("should handle langPrefix mode correctly", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ path: ["fr", "test", "page"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getPage).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "en", "/fr/test/page");
            expect(result).toBeDefined();
        });
        it("should redirect to default locale when locale not in langs", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ path: ["de", "test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
        });
        it("should call getLocalizedPath for non-default locale", async () => {
            vi.mocked(API.getLocalizedPath).mockResolvedValue("/fr/test");
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "fr", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getLocalizedPath).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, mockConfig.langs, "fr", "/test", "en");
        });
        it("should not call getLocalizedPath for default locale", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "en", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getLocalizedPath).not.toHaveBeenCalled();
        });
    });
    describe("Path Handling", () => {
        it("should handle root path correctly", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "en", path: [""] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getPage).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "en", "/");
        });
        it("should handle nested paths correctly", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({
                    locale: "en",
                    path: ["about", "team", "members"],
                }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getPage).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "en", "/about/team/members");
        });
        it("should handle path with empty segments", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "en", path: [""] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(fetchLinks).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "en", ["/"]);
        });
    });
    describe("Layout and Page Validation", () => {
        it("should call notFound when no layout exists", async () => {
            vi.mocked(fetchLinks).mockResolvedValue([]);
            await expect(EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            })).rejects.toThrow("notFound");
        });
        it("should call notFound when layout page is undefined", async () => {
            vi.mocked(fetchLinks).mockResolvedValue([{ page: undefined }]);
            await expect(EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            })).rejects.toThrow("notFound");
        });
        it("should call notFound when menuPath is inactive", async () => {
            vi.mocked(Breadcrumb.reverse).mockReturnValue({
                ...mockMenuPath,
                active: false,
            });
            await expect(EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            })).rejects.toThrow("notFound");
        });
        it("should call notFound when menuPath is null", async () => {
            vi.mocked(Breadcrumb.reverse).mockReturnValue(null);
            await expect(EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            })).rejects.toThrow("notFound");
        });
    });
    describe("Data Fetching", () => {
        it("should fetch links with correct parameters", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "fr", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(fetchLinks).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "fr", expect.any(Array));
        });
        it("should fetch page data with correct parameters", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "fr", path: ["about"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getPage).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "fr", "/about");
        });
        it("should fetch language switches with correct parameters", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "en", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(API.getLangSwitches).toHaveBeenCalledWith(mockConfig.edenApi, mockConfig.site, "en", mockConfig.langs, "/");
        });
        it("should handle page without seo data", async () => {
            vi.mocked(API.getPage).mockResolvedValue({
                zones: { main: "content" },
                breadcrumb: mockBreadcrumb,
            });
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
        });
        it("should handle page without zones", async () => {
            vi.mocked(API.getPage).mockResolvedValue({
                seo: { title: "Test" },
                breadcrumb: mockBreadcrumb,
            });
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
        });
        it("should handle null page data gracefully", async () => {
            vi.mocked(API.getPage).mockResolvedValue(null);
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
        });
    });
    describe("Breadcrumb Processing", () => {
        it("should deep clone breadcrumb", async () => {
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(Breadcrumb.deepClone).toHaveBeenCalledWith(mockBreadcrumb);
        });
        it("should reverse breadcrumb to get menuPath", async () => {
            const clonedBreadcrumb = { ...mockBreadcrumb };
            vi.mocked(Breadcrumb.deepClone).mockReturnValue(clonedBreadcrumb);
            await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(Breadcrumb.reverse).toHaveBeenCalledWith(clonedBreadcrumb);
        });
    });
    describe("Context and Config", () => {
        it("should create correct edenContext", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "en", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            render(result);
            const contextProvider = screen.getByTestId("context-provider");
            const contextData = JSON.parse(contextProvider.getAttribute("data-context") || "{}");
            expect(contextData.edenApi).toBe(mockConfig.edenApi);
            expect(contextData.site).toBe(mockConfig.site);
            expect(contextData.lang).toBe("en");
            expect(contextData.path).toBe("/test");
        });
        it("should create correct config object with lang", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "fr", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            render(result);
            const contextProvider = screen.getByTestId("context-provider");
            const contextData = JSON.parse(contextProvider.getAttribute("data-context") || "{}");
            expect(contextData.lang).toBe("fr");
            expect(contextData.config.site).toBe(mockConfig.site);
            expect(contextData.config.langs).toEqual(mockConfig.langs);
        });
        it("should include domain in context when available", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "en", path: ["test"] }),
                children: _jsx("div", { children: "Test" }),
            });
            render(result);
            const contextProvider = screen.getByTestId("context-provider");
            const contextData = JSON.parse(contextProvider.getAttribute("data-context") || "{}");
            expect(contextData.domain).toBe("example.com");
        });
        it("should include metadata from menuPath", async () => {
            const menuPathWithMetadata = {
                ...mockMenuPath,
                metadata: { customKey: "customValue", numericKey: 42 },
            };
            vi.mocked(Breadcrumb.reverse).mockReturnValue(menuPathWithMetadata);
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            render(result);
            const contextProvider = screen.getByTestId("context-provider");
            const contextData = JSON.parse(contextProvider.getAttribute("data-context") || "{}");
            expect(contextData.metadata).toEqual({
                customKey: "customValue",
                numericKey: 42,
            });
        });
        it("should use empty metadata when menuPath has none", async () => {
            const menuPathWithoutMetadata = {
                ...mockMenuPath,
                metadata: undefined,
            };
            vi.mocked(Breadcrumb.reverse).mockReturnValue(menuPathWithoutMetadata);
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            render(result);
            const contextProvider = screen.getByTestId("context-provider");
            const contextData = JSON.parse(contextProvider.getAttribute("data-context") || "{}");
            expect(contextData.metadata).toEqual({});
        });
    });
    describe("Layout Props", () => {
        it("should call getProps function if defined", async () => {
            const getProps = vi.fn().mockResolvedValue({ customProp: "value" });
            const imports = {
                "default-layout": Promise.resolve({
                    default: () => _jsx("div", { children: "Layout" }),
                    getProps,
                }),
            };
            await EdenLayout({
                layouts: ["default-layout"],
                imports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(getProps).toHaveBeenCalledWith(expect.objectContaining({
                layout: "default-layout",
                edenApi: mockConfig.edenApi,
                site: mockConfig.site,
            }));
        });
        it("should handle layout without getProps function", async () => {
            const imports = {
                "default-layout": Promise.resolve({
                    default: () => _jsx("div", { children: "Layout" }),
                }),
            };
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
        });
        it("should include pageProps in context", async () => {
            const customPageProps = { title: "Custom Title", items: [1, 2, 3] };
            const imports = {
                "default-layout": Promise.resolve({
                    default: () => _jsx("div", { children: "Layout" }),
                    getProps: async () => customPageProps,
                }),
            };
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            render(result);
            const contextProvider = screen.getByTestId("context-provider");
            const contextData = JSON.parse(contextProvider.getAttribute("data-context") || "{}");
            expect(contextData.pageProps).toEqual(customPageProps);
        });
    });
    describe("HTML Structure", () => {
        it("should create layout with correct locale", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve({ locale: "fr", path: [""] }),
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
            render(result);
            const contextProvider = screen.getByTestId("context-provider");
            const contextData = JSON.parse(contextProvider.getAttribute("data-context") || "{}");
            expect(contextData.lang).toBe("fr");
        });
        it("should apply className when provided", async () => {
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                className: "custom-theme dark-mode",
                children: _jsx("div", { children: "Test" }),
            });
            expect(result).toBeDefined();
            render(result);
            expect(screen.getByTestId("context-provider")).toBeInTheDocument();
        });
        it("should include children in rendered output", async () => {
            const testChild = _jsx("div", { "data-testid": "test-content", children: "Test Content" });
            const result = await EdenLayout({
                layouts: ["default-layout"],
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: testChild,
            });
            expect(result).toBeDefined();
            render(result);
            expect(screen.getByTestId("context-provider")).toBeInTheDocument();
        });
    });
    describe("Layouts Formatting", () => {
        it("should format layouts array correctly", async () => {
            const layoutsArray = ["default", "custom", "special"];
            await EdenLayout({
                layouts: layoutsArray,
                imports: mockImports,
                params: Promise.resolve(mockParams),
                children: _jsx("div", { children: "Test" }),
            });
            expect(Layouts.format).toHaveBeenCalledWith(layoutsArray);
        });
    });
});
describe("generateStaticParams", () => {
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(getConfig).mockReturnValue({
            edenApi: "https://api.test.com",
            site: "test-site",
            langs: ["en", "fr"],
            domains: {},
            langPrefix: false,
            siteName: "Test Site",
            preProd: false,
        });
    });
    it("should generate static params based on module config", async () => {
        const exportPathMap = {
            "/en": {},
            "/en/about": {},
            "/fr": {},
            "/fr/a-propos": {},
        };
        vi.mocked(API.getExportPathMap).mockResolvedValue(exportPathMap);
        const result = await generateStaticParams();
        expect(API.getExportPathMap).toHaveBeenCalledWith("https://api.test.com", "test-site", "en,fr");
        expect(result).toEqual([
            { locale: "en", path: [] },
            { locale: "en", path: ["about"] },
            { locale: "fr", path: [] },
            { locale: "fr", path: ["a-propos"] },
        ]);
    });
    it("should handle root path correctly", async () => {
        const exportPathMap = {
            "/en": {},
        };
        vi.mocked(API.getExportPathMap).mockResolvedValue(exportPathMap);
        const result = await generateStaticParams();
        expect(result).toEqual([{ locale: "en", path: [] }]);
    });
    it("should handle nested paths", async () => {
        const exportPathMap = {
            "/en/about/team/members": {},
        };
        vi.mocked(API.getExportPathMap).mockResolvedValue(exportPathMap);
        const result = await generateStaticParams();
        expect(result).toEqual([
            { locale: "en", path: ["about", "team", "members"] },
        ]);
    });
    it("should handle empty exportPathMap", async () => {
        vi.mocked(API.getExportPathMap).mockResolvedValue({});
        const result = await generateStaticParams();
        expect(result).toEqual([]);
    });
});
describe("dynamicParams", () => {
    it("should be true", () => {
        expect(dynamicParams).toBe(true);
    });
});
