import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render } from "@testing-library/react";
import EdenHead from "../head";
import * as Context from "../context";
vi.mock("../context", async () => {
    const actual = await vi.importActual("../context");
    return {
        ...actual,
        useEdenContext: vi.fn(),
    };
});
describe("EdenHead", () => {
    const mockContext = {
        lang: "en",
        domain: "test.com",
        edenMode: null,
        config: {
            langs: ["en", "fr"],
            domains: { en: "test.com", fr: "test.fr" },
            preProd: false,
            siteName: "Test Site",
            edenApi: "",
            site: "",
            langPrefix: false,
            layout: "",
            layouts: [],
        },
        menuPath: {
            id: 1,
            active: true,
            disabled: false,
            name: "home",
            title: "Home Page",
            subTitle: "Welcome",
            path: "/",
            page: "home",
            prev: null,
            next: null,
            menu: "main",
        },
        seo: {
            title: "Custom Title",
            description: "Custom Description",
            robots: "index,follow",
            canonical: "https://test.com/custom",
            keywords: "test,keywords",
        },
        langSwitches: {
            fr: { path: "/fr", page: "home", disabled: false },
        },
        pageProps: {},
        zones: {},
        isConnected: false,
        edenApi: "",
        site: "",
        path: "",
        breadcrumb: {},
        metadata: {},
    };
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
        document.head.innerHTML = "";
    });
    it("should render title with custom SEO title", () => {
        render(_jsx(EdenHead, {}));
        const title = document.head.querySelector("title");
        expect(title).toBeTruthy();
        expect(title?.textContent).toBe("Custom Title");
    });
    it("should render title with page title and site name", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            seo: {},
        });
        render(_jsx(EdenHead, {}));
        const title = document.head.querySelector("title");
        expect(title).toBeTruthy();
        expect(title?.textContent).toBe("Home Page Welcome - Test Site");
    });
    it("should render title without subtitle", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            menuPath: { ...mockContext.menuPath, subTitle: undefined },
            seo: {},
        });
        render(_jsx(EdenHead, {}));
        const title = document.head.querySelector("title");
        expect(title).toBeTruthy();
        expect(title?.textContent).toBe("Home Page - Test Site");
    });
    it("should render description meta tag", () => {
        render(_jsx(EdenHead, {}));
        const meta = document.head.querySelector('meta[name="description"]');
        expect(meta?.getAttribute("content")).toBe("Custom Description");
    });
    it("should render robots meta tag from SEO", () => {
        render(_jsx(EdenHead, {}));
        const meta = document.head.querySelector('meta[name="robots"]');
        expect(meta?.getAttribute("content")).toBe("index,follow");
    });
    it("should render noindex meta tag when page is disabled", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            menuPath: { ...mockContext.menuPath, disabled: true },
        });
        render(_jsx(EdenHead, {}));
        const metaTags = document.head.querySelectorAll('meta[name="robots"]');
        expect(metaTags.length).toBeGreaterThan(0);
        const noindexTag = Array.from(metaTags).find((tag) => tag.getAttribute("content") === "noindex, nofollow");
        expect(noindexTag).toBeTruthy();
    });
    it("should render noindex meta tag when in pre-prod mode", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            config: { ...mockContext.config, preProd: true },
        });
        render(_jsx(EdenHead, {}));
        const metaTags = document.head.querySelectorAll('meta[name="robots"]');
        const noindexTag = Array.from(metaTags).find((tag) => tag.getAttribute("content") === "noindex, nofollow");
        expect(noindexTag).toBeTruthy();
    });
    it("should render noindex meta tag for specific languages in pre-prod", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            lang: "fr",
            config: { ...mockContext.config, preProd: ["fr"] },
        });
        render(_jsx(EdenHead, {}));
        const metaTags = document.head.querySelectorAll('meta[name="robots"]');
        const noindexTag = Array.from(metaTags).find((tag) => tag.getAttribute("content") === "noindex, nofollow");
        expect(noindexTag).toBeTruthy();
    });
    it("should not render noindex when not in pre-prod language", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            lang: "en",
            config: { ...mockContext.config, preProd: ["fr"] },
        });
        render(_jsx(EdenHead, {}));
        const metaTags = document.head.querySelectorAll('meta[name="robots"]');
        const noindexTag = Array.from(metaTags).find((tag) => tag.getAttribute("content") === "noindex, nofollow");
        expect(noindexTag).toBeFalsy();
    });
    it("should render notranslate meta tag when in edenMode", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            edenMode: "editing",
        });
        render(_jsx(EdenHead, {}));
        const meta = document.head.querySelector('meta[name="google"]');
        expect(meta?.getAttribute("content")).toBe("notranslate");
    });
    it("should not render notranslate meta tag when not in edenMode", () => {
        render(_jsx(EdenHead, {}));
        const meta = document.head.querySelector('meta[name="google"]');
        expect(meta).toBeFalsy();
    });
    it("should render format-detection meta tag", () => {
        render(_jsx(EdenHead, {}));
        const meta = document.head.querySelector('meta[name="format-detection"]');
        expect(meta?.getAttribute("content")).toBe("telephone=no, date=no, email=no, address=no");
    });
    it("should render canonical link tag from SEO", () => {
        render(_jsx(EdenHead, {}));
        const link = document.head.querySelector('link[rel="canonical"]');
        expect(link?.getAttribute("href")).toBe("https://test.com/custom");
    });
    it("should render default canonical link tag", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            menuPath: { ...mockContext.menuPath, path: "/about" },
            seo: {},
        });
        render(_jsx(EdenHead, {}));
        const link = document.head.querySelector('link[rel="canonical"]');
        expect(link?.getAttribute("href")).toBe("https://test.com/about/");
    });
    it("should render canonical without trailing slash for index", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            seo: {},
        });
        render(_jsx(EdenHead, {}));
        const link = document.head.querySelector('link[rel="canonical"]');
        expect(link?.getAttribute("href")).toBe("https://test.com/");
    });
    it("should not render canonical when domain is missing", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            domain: undefined,
            seo: {},
        });
        render(_jsx(EdenHead, {}));
        const link = document.head.querySelector('link[rel="canonical"]');
        expect(link).toBeFalsy();
    });
    it("should render alternate language links", () => {
        render(_jsx(EdenHead, {}));
        const alternateLinks = document.head.querySelectorAll('link[rel="alternate"]');
        expect(alternateLinks.length).toBe(1);
        const frLink = Array.from(alternateLinks).find((link) => link.getAttribute("hrefLang") === "fr");
        expect(frLink?.getAttribute("href")).toBe("https://test.fr/fr/");
    });
    it("should not render alternate link when disabled", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            langSwitches: {
                fr: { path: "/fr", page: "home", disabled: true },
            },
        });
        render(_jsx(EdenHead, {}));
        const alternateLinks = document.head.querySelectorAll('link[rel="alternate"]');
        expect(alternateLinks.length).toBe(0);
    });
    it("should not render alternate link when domain is missing", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            config: {
                ...mockContext.config,
                domains: { en: "test.com" },
            },
        });
        render(_jsx(EdenHead, {}));
        const alternateLinks = document.head.querySelectorAll('link[rel="alternate"]');
        expect(alternateLinks.length).toBe(0);
    });
    it("should not render alternate link when path is missing", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            langSwitches: {
                fr: { path: "", page: "home", disabled: false },
            },
        });
        render(_jsx(EdenHead, {}));
        const alternateLinks = document.head.querySelectorAll('link[rel="alternate"]');
        expect(alternateLinks.length).toBe(0);
    });
    it("should render alternate link with trailing slash for non-home pages", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            langSwitches: {
                fr: { path: "/fr/about", page: "about", disabled: false },
            },
        });
        render(_jsx(EdenHead, {}));
        const alternateLinks = document.head.querySelectorAll('link[rel="alternate"]');
        const frLink = Array.from(alternateLinks).find((link) => link.getAttribute("hrefLang") === "fr");
        expect(frLink?.getAttribute("href")).toBe("https://test.fr/fr/about/");
    });
    it("should strip HTML tags from title", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            menuPath: {
                ...mockContext.menuPath,
                title: "<strong>Bold</strong> Title",
                subTitle: "<em>Italic</em> Subtitle",
            },
            seo: {},
        });
        render(_jsx(EdenHead, {}));
        const title = document.head.querySelector("title");
        expect(title).toBeTruthy();
        expect(title?.textContent).toBe("Bold Title Italic Subtitle - Test Site");
    });
    it("should use page name when title is missing", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            menuPath: {
                ...mockContext.menuPath,
                title: "",
                name: "home-page",
                subTitle: undefined,
            },
            seo: {},
        });
        render(_jsx(EdenHead, {}));
        const title = document.head.querySelector("title");
        expect(title).toBeTruthy();
        expect(title?.textContent).toBe("home-page - Test Site");
    });
});
