import { jsx as _jsx, jsxs as _jsxs } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, screen } from "@testing-library/react";
import EdenContextProvider, { useEdenContext } from "../context";
describe("EdenContextProvider", () => {
    const mockContextValue = {
        pageProps: { test: "value" },
        config: {
            edenApi: "https://api.test.com",
            site: "test-site",
            lang: "en",
            langs: ["en", "fr"],
            domains: { en: "test.com", fr: "test.fr" },
            siteName: "Test Site",
            preProd: false,
            layout: "default",
            layouts: ["default", "custom"],
            langPrefix: false,
        },
        zones: { header: "<div>Header</div>", footer: "<div>Footer</div>" },
        isConnected: false,
        edenApi: "https://api.test.com",
        site: "test-site",
        lang: "en",
        path: "/test-path",
        edenMode: null,
        menuPath: {
            id: 1,
            active: true,
            disabled: false,
            highlight: false,
            name: "home",
            title: "Home",
            subTitle: "Welcome",
            shortTitle: "Home",
            menu: "main",
            path: "/",
            page: "home-page",
            prev: null,
            next: null,
            metadata: { key: "value" },
        },
        breadcrumb: {
            id: 1,
            active: true,
            disabled: false,
            highlight: false,
            name: "home",
            title: "Home",
            subTitle: "Welcome",
            shortTitle: "Home",
            menu: "main",
            path: "/",
            page: "home-page",
            prev: null,
            next: null,
            metadata: { key: "value" },
        },
        seo: {
            title: "Test Title",
            description: "Test Description",
            robots: "index,follow",
            canonical: "https://test.com/",
            keywords: "test,keywords",
        },
        metadata: { author: "Test Author" },
        langSwitches: { fr: { path: "/fr/test", page: "test", disabled: false } },
    };
    beforeEach(() => {
        vi.clearAllMocks();
        window.setEdenMode = vi.fn();
        window.updateEdenDraftMode = vi.fn();
    });
    it("should render children", () => {
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx("div", { children: "Test Child" }) }));
        expect(screen.getByText("Test Child")).toBeInTheDocument();
    });
    it("should provide context values", () => {
        const TestComponent = () => {
            const context = useEdenContext();
            return _jsx("div", { children: context.site });
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByText("test-site")).toBeInTheDocument();
    });
    it("should initialize isConnected state to false", () => {
        const TestComponent = () => {
            const { isConnected } = useEdenContext();
            return _jsx("div", { children: isConnected ? "Connected" : "Not Connected" });
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByText("Not Connected")).toBeInTheDocument();
    });
    it("should initialize edenMode state to null", () => {
        const TestComponent = () => {
            const { edenMode } = useEdenContext();
            return _jsx("div", { children: edenMode || "No Mode" });
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByText("No Mode")).toBeInTheDocument();
    });
    it("should provide access to all context properties", () => {
        const TestComponent = () => {
            const context = useEdenContext();
            return (_jsxs("div", { children: [_jsx("span", { "data-testid": "lang", children: context.lang }), _jsx("span", { "data-testid": "path", children: context.path }), _jsx("span", { "data-testid": "site", children: context.site }), _jsx("span", { "data-testid": "menuPath", children: context.menuPath.title })] }));
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByTestId("lang")).toHaveTextContent("en");
        expect(screen.getByTestId("path")).toHaveTextContent("/test-path");
        expect(screen.getByTestId("site")).toHaveTextContent("test-site");
        expect(screen.getByTestId("menuPath")).toHaveTextContent("Home");
    });
    it("should provide config object", () => {
        const TestComponent = () => {
            const { config } = useEdenContext();
            return (_jsxs("div", { children: [_jsx("span", { "data-testid": "config-lang", children: config.lang }), _jsx("span", { "data-testid": "config-siteName", children: config.siteName })] }));
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByTestId("config-lang")).toHaveTextContent("en");
        expect(screen.getByTestId("config-siteName")).toHaveTextContent("Test Site");
    });
    it("should provide zones object", () => {
        const TestComponent = () => {
            const { zones } = useEdenContext();
            return (_jsx("div", { children: _jsx("span", { "data-testid": "zone-count", children: Object.keys(zones).length }) }));
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByTestId("zone-count")).toHaveTextContent("2");
    });
    it("should provide seo object", () => {
        const TestComponent = () => {
            const { seo } = useEdenContext();
            return (_jsxs("div", { children: [_jsx("span", { "data-testid": "seo-title", children: seo.title }), _jsx("span", { "data-testid": "seo-description", children: seo.description })] }));
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByTestId("seo-title")).toHaveTextContent("Test Title");
        expect(screen.getByTestId("seo-description")).toHaveTextContent("Test Description");
    });
    it("should provide metadata object", () => {
        const TestComponent = () => {
            const { metadata } = useEdenContext();
            return (_jsx("div", { children: _jsx("span", { "data-testid": "metadata-author", children: metadata.author }) }));
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByTestId("metadata-author")).toHaveTextContent("Test Author");
    });
    it("should provide langSwitches object", () => {
        const TestComponent = () => {
            const { langSwitches } = useEdenContext();
            return (_jsx("div", { children: _jsx("span", { "data-testid": "lang-switch-fr", children: langSwitches.fr?.path }) }));
        };
        render(_jsx(EdenContextProvider, { value: mockContextValue, children: _jsx(TestComponent, {}) }));
        expect(screen.getByTestId("lang-switch-fr")).toHaveTextContent("/fr/test");
    });
});
