import { errors } from "./error.js";
import { isLocalhost } from "./util.js";
const token = window.sessionStorage.getItem("eden_token");
function fetcher(host, method = "GET") {
    return async (url, { resultType = "text", headers, body, formData = false } = {}) => {
        try {
            const response = await fetch(`${host}${url}`, {
                method,
                headers,
                body: formData
                    ? body
                    : method !== "GET"
                        ? JSON.stringify(body)
                        : undefined,
            });
            if (response.ok) {
                return resultType === "text"
                    ? await response.text()
                    : await response.json();
            }
            else {
                throw errors[response.status] || errors["500"];
            }
        }
        catch (e) {
            if (isLocalhost()) {
                console.info("error", e);
            }
            return { error: e };
        }
    };
}
const fetchRelativeGet = fetcher("");
const fetchApiPost = fetcher(window.EdenConfig.edenApi, "POST");
const fetchApiGet = fetcher(window.EdenConfig.edenApi);
export function fetchFormats() {
    const { templates } = window.EdenConfig;
    return fetchRelativeGet(templates, {
        resultType: "text",
    });
}
export async function fetchRevalidateAll(lang) {
    const { langs, langPrefix, domains } = window.EdenConfig;
    const { default: EdenRevalidationIndicator } = await import("../user-interface/page/eden-revalidation-indicator.js");
    const revalidationIndicator = new EdenRevalidationIndicator();
    revalidationIndicator.init();
    if (langPrefix || isLocalhost()) {
        const locale = lang || langs.join(",");
        await fetchRelativeGet(`/api/revalidate/?token=${token}&all=true&lang=${locale}`, { resultType: "json" });
    }
    else {
        for await (const locale of langs) {
            const domain = domains[locale];
            if (domain) {
                await fetchRelativeGet(`https://${domain}/api/revalidate/?token=${token}&all=true&lang=${locale}`, { resultType: "json" });
            }
        }
    }
    revalidationIndicator.done();
}
export function fetchRevalidate(lang, path) {
    const { lang: currentLang, domains } = window.EdenConfig;
    if (lang === currentLang) {
        return fetchRelativeGet(`/api/revalidate/?token=${token}&path=${path}`, {
            resultType: "json",
        });
    }
    const domain = domains[lang];
    return fetchRelativeGet(`https://${domain}/api/revalidate/?token=${token}&path=${path}`);
}
export function sendRevalidateAllBeacon(lang) {
    const { langs, langPrefix, domains } = window.EdenConfig;
    if (lang || langPrefix || isLocalhost()) {
        navigator.sendBeacon(`/api/revalidate/?token=${token}&all=true&lang=${lang || langs.join(",")}`);
    }
    else {
        for (const locale of langs) {
            const domain = domains[locale];
            if (domain) {
                navigator.sendBeacon(`https://${domain}/api/revalidate/?token=${token}&all=true&lang=${locale}`);
            }
        }
    }
}
export function fetchStructure(path, lang, langPrefix) {
    const { site } = window.EdenConfig;
    return fetchApiGet(`/api/admin/structure?path=${path}&lang=${lang}&site=${site}&langPrefix=${langPrefix}`, {
        resultType: "json",
        headers: { "Content-Type": "application/json", authorization: token },
    });
}
export function fetchClearCache() {
    const { site } = window.EdenConfig;
    return fetchApiGet(`/api/clear-cache?site=${site}`, {
        resultType: "json",
        headers: { "Content-Type": "application/json" },
    });
}
export function fetchNode(id) {
    const { site } = window.EdenConfig;
    return fetchApiGet(`/api/admin/structure/node?navItemId=${id}&site=${site}`, {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
    });
}
export function fetchNodeCreation(node) {
    const { site } = window.EdenConfig;
    const { email } = window.Eden.user;
    return fetchApiPost("/api/admin/structure/create-node", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: { site, node, user: email },
    });
}
export function fetchNodeUpdate(node) {
    const { site } = window.EdenConfig;
    const { email } = window.Eden.user;
    return fetchApiPost("/api/admin/structure/save-node", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: { site, node, user: email },
    });
}
export function fetchNodeMove(node) {
    const { site } = window.EdenConfig;
    const { email } = window.Eden.user;
    return fetchApiPost("/api/admin/structure/move-node", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: { site, node, user: email },
    });
}
export function fetchNodeRemoval(node) {
    const { site } = window.EdenConfig;
    const { email } = window.Eden.user;
    return fetchApiPost("/api/admin/structure/remove-node", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: { site, node, user: email },
    });
}
export function fetchNodeRename(node) {
    const { site } = window.EdenConfig;
    const { email } = window.Eden.user;
    return fetchApiPost("/api/admin/structure/rename-node", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: { site, node, user: email },
    });
}
export function fetchContentSave(lang, path, zones, startedAt) {
    const { email: user } = window.Eden.user;
    const { site, edenMode, langPrefix } = window.EdenConfig;
    return fetchApiPost("/api/admin/content/save", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            edenMode,
            site,
            lang,
            path,
            zones,
            langPrefix,
            user,
            startedAt,
        },
    });
}
export function fetchLastUpdated(lang, langPrefix, path) {
    const { site } = window.EdenConfig;
    return fetchApiPost(`/api/admin/content/get-last-updated`, {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: { site, lang, langPrefix, path },
    });
}
export function fetchTimestamp() {
    return fetchApiGet(`/api/admin/timestamp`, {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
    });
}
export const FILE_MANAGER_EXTENSIONS = ["jpg", "jpeg", "png"];
export function fetchFileManagerList(path) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/list", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            path,
            extensions: FILE_MANAGER_EXTENSIONS,
        },
    });
}
export function fetchFileManagerStorageStats() {
    const { site } = window.EdenConfig;
    return fetchApiGet(`/api/files/storage/${site}`, {
        resultType: "json",
        headers: {
            authorization: token,
        },
    });
}
export function createFileManagerFolder(name, path) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/createDir", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            name,
            path,
            extensions: FILE_MANAGER_EXTENSIONS,
        },
    });
}
export function removeFileManagerItem(path, paths) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/remove", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            path,
            paths,
            extensions: FILE_MANAGER_EXTENSIONS,
        },
    });
}
export function renameFileManagerItem(file, name) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/rename", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            file,
            name,
            extensions: FILE_MANAGER_EXTENSIONS,
        },
    });
}
export function cropFileManagerItem(path, left, top, width, height, targetWidth, targetHeight, only) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/crop", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            path,
            left,
            top,
            width,
            height,
            targetWidth,
            targetHeight,
            only,
        },
    });
}
export function moveFileManagerItem(path, paths, target) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/move", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            path,
            paths,
            target,
            extensions: FILE_MANAGER_EXTENSIONS,
        },
    });
}
export function fileManagerMoveList(paths) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/moveList", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            paths,
        },
    });
}
export function rotateFileManagerImage(path) {
    const { site } = window.EdenConfig;
    return fetchApiPost("/api/files/rotate", {
        resultType: "json",
        headers: {
            "Content-Type": "application/json",
            authorization: token,
        },
        body: {
            site,
            path,
            extensions: FILE_MANAGER_EXTENSIONS,
        },
    });
}
export function uploadFileManagerItem(payload) {
    return fetchApiPost("/api/files/upload", {
        resultType: "json",
        headers: {
            authorization: token,
        },
        body: payload,
        formData: true,
    });
}
