import { describe, it, expect } from "vitest";
import { parseUrl, buildParams } from "../url";
describe("URL Service", () => {
    describe("parseUrl", () => {
        it("should parse a complete URL", () => {
            const result = parseUrl("https://example.com:8080/path?key=value#hash");
            expect(result).toBeDefined();
            expect(result?.protocol).toBe("https:");
            expect(result?.hostname).toBe("example.com");
            expect(result?.port).toBe("8080");
            expect(result?.path).toBe("/path");
            expect(result?.search).toBe("?key=value");
            expect(result?.hash).toBe("#hash");
        });
        it("should parse URL with query parameters", () => {
            const result = parseUrl("https://example.com?foo=bar&baz=qux");
            expect(result?.searchObject).toEqual({
                foo: "bar",
                baz: "qux",
            });
        });
        it("should handle URL without query parameters", () => {
            const result = parseUrl("https://example.com/path");
            expect(result?.searchObject).toEqual({});
            expect(result?.search).toBe("");
        });
        it("should handle URL with port", () => {
            const result = parseUrl("http://localhost:3000/api");
            expect(result?.host).toBe("localhost:3000");
            expect(result?.port).toBe("3000");
        });
        it("should handle URL with hash", () => {
            const result = parseUrl("https://example.com#section");
            expect(result?.hash).toBe("#section");
        });
        it("should return undefined for non-http URLs", () => {
            const result = parseUrl("/relative/path");
            expect(result).toBeUndefined();
        });
        it("should return undefined for empty string", () => {
            const result = parseUrl("");
            expect(result).toBeUndefined();
        });
        it("should handle URL with multiple query parameters", () => {
            const result = parseUrl("https://example.com?param1=value1&param2=value2&param3=value3");
            expect(result?.searchObject).toEqual({
                param1: "value1",
                param2: "value2",
                param3: "value3",
            });
        });
        it("should handle URL with empty query parameter values", () => {
            const result = parseUrl("https://example.com?key=");
            expect(result?.searchObject).toEqual({
                key: "",
            });
        });
        it("should preserve original URL", () => {
            const url = "https://example.com:8080/path?key=value#hash";
            const result = parseUrl(url);
            expect(result?.url).toBe(url);
        });
        it("should handle https URLs", () => {
            const result = parseUrl("https://secure.example.com");
            expect(result?.protocol).toBe("https:");
        });
        it("should handle http URLs", () => {
            const result = parseUrl("http://example.com");
            expect(result?.protocol).toBe("http:");
        });
    });
    describe("buildParams", () => {
        it("should build query string from params", () => {
            const result = buildParams({ foo: "bar", baz: "qux" }, []);
            expect(result).toBe("?rel=0&foo=bar&baz=qux");
        });
        it("should include default rel parameter", () => {
            const result = buildParams({}, []);
            expect(result).toBe("?rel=0");
        });
        it("should exclude specified parameters", () => {
            const result = buildParams({ foo: "bar", baz: "qux" }, ["baz"]);
            expect(result).toContain("foo=bar");
            expect(result).not.toContain("baz=qux");
        });
        it("should handle empty params", () => {
            const result = buildParams(undefined, []);
            expect(result).toBe("?rel=0");
        });
        it("should override default params", () => {
            const result = buildParams({ rel: "1" }, []);
            expect(result).toBe("?rel=1");
        });
        it("should exclude rel when specified", () => {
            const result = buildParams({ foo: "bar" }, ["rel"]);
            expect(result).toBe("?foo=bar");
            expect(result).not.toContain("rel");
        });
        it("should handle falsy parameter values", () => {
            const result = buildParams({ foo: "", bar: "0" }, []);
            expect(result).not.toContain("foo=");
            expect(result).toContain("bar=0");
        });
        it("should use & separator for subsequent parameters", () => {
            const result = buildParams({ a: "1", b: "2", c: "3" }, []);
            expect(result).toMatch(/\?.*&.*&.*/);
        });
        it("should handle single parameter", () => {
            const result = buildParams({ foo: "bar" }, ["rel"]);
            expect(result).toBe("?foo=bar");
        });
        it("should handle multiple exclusions", () => {
            const result = buildParams({ a: "1", b: "2", c: "3", d: "4" }, [
                "b",
                "d",
            ]);
            expect(result).toContain("a=1");
            expect(result).toContain("c=3");
            expect(result).not.toContain("b=2");
            expect(result).not.toContain("d=4");
        });
    });
});
