import { describe, it, expect } from "vitest";
import { normalize, slugify } from "../text";
describe("Text Service", () => {
    describe("normalize", () => {
        it("should remove diacritics from text", () => {
            expect(normalize("café")).toBe("cafe");
            expect(normalize("naïve")).toBe("naive");
            expect(normalize("résumé")).toBe("resume");
        });
        it("should handle accented characters", () => {
            expect(normalize("àèéêëïîôùü").length).toBeGreaterThanOrEqual(9);
            expect(normalize("ÀÈÉÊËÏÎÔÙÜ").length).toBeGreaterThanOrEqual(9);
        });
        it("should handle text without diacritics", () => {
            expect(normalize("hello")).toBe("hello");
            expect(normalize("world")).toBe("world");
            expect(normalize("123")).toBe("123");
        });
        it("should handle empty string", () => {
            expect(normalize("")).toBe("");
        });
        it("should handle special characters", () => {
            expect(normalize("ñ")).toBeTruthy();
            expect(normalize("ç")).toBeTruthy();
            expect(normalize("ø")).toBeTruthy();
        });
        it("should preserve spaces and punctuation", () => {
            expect(normalize("Hello, World!")).toBe("Hello, World!");
            expect(normalize("café au lait")).toBe("cafe au lait");
        });
    });
    describe("slugify", () => {
        it("should convert text to slug format", () => {
            expect(slugify("Hello World")).toBe("hello-world");
            expect(slugify("This is a Test")).toBe("this-is-a-test");
        });
        it("should remove diacritics", () => {
            expect(slugify("café")).toBe("cafe");
            expect(slugify("Crème Brûlée")).toBe("creme-brulee");
        });
        it("should replace spaces with hyphens", () => {
            expect(slugify("one two three")).toBe("one-two-three");
            expect(slugify("multiple   spaces")).toBe("multiple-spaces");
        });
        it("should remove special characters", () => {
            expect(slugify("hello@world!")).toBe("helloworld");
            expect(slugify("test#123$456")).toBe("test123456");
            expect(slugify("foo&bar")).toBe("foobar");
        });
        it("should convert to lowercase", () => {
            expect(slugify("UPPERCASE")).toBe("uppercase");
            expect(slugify("MixedCase")).toBe("mixedcase");
        });
        it("should handle multiple consecutive hyphens", () => {
            expect(slugify("hello---world")).toBe("hello-world");
            expect(slugify("test--slug")).toBe("test-slug");
        });
        it("should trim whitespace", () => {
            expect(slugify("  hello world  ")).toBe("hello-world");
            expect(slugify("\n\ttest\n\t")).toBe("test");
        });
        it("should handle empty string", () => {
            expect(slugify("")).toBe("");
        });
        it("should preserve numbers and underscores", () => {
            expect(slugify("test_123")).toBe("test_123");
            expect(slugify("item_456")).toBe("item_456");
        });
        it("should handle complex slugs", () => {
            expect(slugify("JavaScript & TypeScript: A Guide!")).toBe("javascript-typescript-a-guide");
            expect(slugify("Développement Web Moderne")).toBe("developpement-web-moderne");
        });
        it("should handle only special characters", () => {
            expect(slugify("@#$%")).toBe("");
            expect(slugify("!!!")).toBe("");
        });
        it("should handle numbers", () => {
            expect(slugify("123")).toBe("123");
            expect(slugify("version 2.0")).toBe("version-20");
        });
    });
});
