import { describe, it, expect, vi } from "vitest";
import { getTextLengthHerlperContent, updateErrorAttribute, preventDefaultIfNecessary, } from "../text-length";
describe("Text Length Service", () => {
    describe("getTextLengthHerlperContent", () => {
        it("should return content with current and max length", () => {
            const element = document.createElement("div");
            element.textContent = "Hello World";
            const options = {
                textLength: { max: 100 },
            };
            const result = getTextLengthHerlperContent(element, options);
            expect(result).toBeDefined();
            expect(result?.content).toContain("11");
            expect(result?.content).toContain("100");
        });
        it("should calculate optimal word count in tooltip", () => {
            const element = document.createElement("div");
            element.textContent = "Test";
            const options = {
                textLength: { max: 50 },
            };
            const result = getTextLengthHerlperContent(element, options);
            expect(result?.tooltip).toContain("10");
        });
        it("should return undefined if no textLength option", () => {
            const element = document.createElement("div");
            const result = getTextLengthHerlperContent(element, {});
            expect(result).toBeUndefined();
        });
        it("should return undefined if options is undefined", () => {
            const element = document.createElement("div");
            const result = getTextLengthHerlperContent(element, undefined);
            expect(result).toBeUndefined();
        });
        it("should handle empty text content", () => {
            const element = document.createElement("div");
            element.textContent = "";
            const options = {
                textLength: { max: 100 },
            };
            const result = getTextLengthHerlperContent(element, options);
            expect(result?.content).toContain("0");
            expect(result?.content).toContain("100");
        });
        it("should handle null text content", () => {
            const element = document.createElement("div");
            const options = {
                textLength: { max: 100 },
            };
            const result = getTextLengthHerlperContent(element, options);
            expect(result?.content).toContain("0");
        });
        it("should format content as HTML spans", () => {
            const element = document.createElement("div");
            element.textContent = "Test";
            const options = {
                textLength: { max: 10 },
            };
            const result = getTextLengthHerlperContent(element, options);
            expect(result?.content).toContain("<span>");
            expect(result?.content).toContain("</span>");
        });
    });
    describe("updateErrorAttribute", () => {
        it("should add error attribute when text exceeds max", () => {
            const element = document.createElement("div");
            element.textContent = "This is a long text";
            const helper = document.createElement("div");
            helper.setAttribute = vi.fn();
            const options = {
                textLength: { max: 10 },
            };
            updateErrorAttribute(element, helper, options);
            expect(helper.setAttribute).toHaveBeenCalledWith("error", "");
        });
        it("should remove error attribute when text is within limit", () => {
            const element = document.createElement("div");
            element.textContent = "Short";
            const helper = document.createElement("div");
            helper.removeAttribute = vi.fn();
            const options = {
                textLength: { max: 100 },
            };
            updateErrorAttribute(element, helper, options);
            expect(helper.removeAttribute).toHaveBeenCalledWith("error");
        });
        it("should do nothing if helper is undefined", () => {
            const element = document.createElement("div");
            const options = {
                textLength: { max: 10 },
            };
            expect(() => updateErrorAttribute(element, undefined, options)).not.toThrow();
        });
        it("should do nothing if options has no textLength", () => {
            const element = document.createElement("div");
            const helper = document.createElement("div");
            expect(() => updateErrorAttribute(element, helper, {})).not.toThrow();
        });
        it("should handle boundary case when text equals max", () => {
            const element = document.createElement("div");
            element.textContent = "1234567890";
            const helper = document.createElement("div");
            helper.setAttribute = vi.fn();
            const options = {
                textLength: { max: 10 },
            };
            updateErrorAttribute(element, helper, options);
            expect(helper.setAttribute).toHaveBeenCalledWith("error", "");
        });
    });
    describe("preventDefaultIfNecessary", () => {
        it("should prevent default when text exceeds max and enforce is true", () => {
            const element = document.createElement("div");
            element.textContent = "This text is too long";
            const event = {
                key: "a",
                preventDefault: vi.fn(),
            };
            const options = {
                textLength: { max: 10 },
            };
            preventDefaultIfNecessary(event, true, element, options);
            expect(event.preventDefault).toHaveBeenCalled();
        });
        it("should not prevent default when text is within limit", () => {
            const element = document.createElement("div");
            element.textContent = "Short";
            const event = {
                key: "a",
                preventDefault: vi.fn(),
            };
            const options = {
                textLength: { max: 100 },
            };
            preventDefaultIfNecessary(event, true, element, options);
            expect(event.preventDefault).not.toHaveBeenCalled();
        });
        it("should not prevent default for Backspace key", () => {
            const element = document.createElement("div");
            element.textContent = "This text is too long";
            const event = {
                key: "Backspace",
                preventDefault: vi.fn(),
            };
            const options = {
                textLength: { max: 10 },
            };
            preventDefaultIfNecessary(event, true, element, options);
            expect(event.preventDefault).not.toHaveBeenCalled();
        });
        it("should not prevent default for Delete key", () => {
            const element = document.createElement("div");
            element.textContent = "This text is too long";
            const event = {
                key: "Delete",
                preventDefault: vi.fn(),
            };
            const options = {
                textLength: { max: 10 },
            };
            preventDefaultIfNecessary(event, true, element, options);
            expect(event.preventDefault).not.toHaveBeenCalled();
        });
        it("should not prevent default for arrow keys", () => {
            const element = document.createElement("div");
            element.textContent = "This text is too long";
            const arrowKeys = ["ArrowLeft", "ArrowRight", "ArrowUp", "ArrowDown"];
            const options = {
                textLength: { max: 10 },
            };
            arrowKeys.forEach((key) => {
                const event = {
                    key,
                    preventDefault: vi.fn(),
                };
                preventDefaultIfNecessary(event, true, element, options);
                expect(event.preventDefault).not.toHaveBeenCalled();
            });
        });
        it("should not prevent default when enforceTextLength is false", () => {
            const element = document.createElement("div");
            element.textContent = "This text is too long";
            const event = {
                key: "a",
                preventDefault: vi.fn(),
            };
            const options = {
                textLength: { max: 10 },
            };
            preventDefaultIfNecessary(event, false, element, options);
            expect(event.preventDefault).not.toHaveBeenCalled();
        });
        it("should do nothing if no textLength option", () => {
            const element = document.createElement("div");
            const event = {
                key: "a",
                preventDefault: vi.fn(),
            };
            preventDefaultIfNecessary(event, true, element, {});
            expect(event.preventDefault).not.toHaveBeenCalled();
        });
    });
});
