import { describe, it, expect, beforeEach } from "vitest";
import { injectStyles } from "../styles";
describe("Styles Service", () => {
    beforeEach(() => {
        document.head.innerHTML = "";
    });
    describe("injectStyles", () => {
        it("should inject styles into document head", () => {
            const id = "test-styles";
            const styles = [".test { color: red; }"];
            injectStyles(id, styles);
            const styleElement = document.getElementById(id);
            expect(styleElement).toBeTruthy();
            expect(styleElement?.tagName).toBe("STYLE");
            expect(styleElement?.innerHTML).toBe(".test { color: red; }");
        });
        it("should join multiple style strings", () => {
            const id = "multi-styles";
            const styles = [
                ".class1 { color: red; }",
                ".class2 { color: blue; }",
                ".class3 { color: green; }",
            ];
            injectStyles(id, styles);
            const styleElement = document.getElementById(id);
            expect(styleElement?.innerHTML).toBe(".class1 { color: red; }.class2 { color: blue; }.class3 { color: green; }");
        });
        it("should update existing styles with same id", () => {
            const id = "update-styles";
            const initialStyles = [".initial { color: red; }"];
            const updatedStyles = [".updated { color: blue; }"];
            injectStyles(id, initialStyles);
            const styleElement1 = document.getElementById(id);
            expect(styleElement1?.innerHTML).toBe(".initial { color: red; }");
            injectStyles(id, updatedStyles);
            const styleElement2 = document.getElementById(id);
            expect(styleElement2).toBe(styleElement1);
            expect(styleElement2?.innerHTML).toBe(".updated { color: blue; }");
        });
        it("should not create duplicate style elements with same id", () => {
            const id = "no-duplicates";
            const styles = [".test { color: red; }"];
            injectStyles(id, styles);
            injectStyles(id, styles);
            injectStyles(id, styles);
            const styleElements = document.querySelectorAll(`#${id}`);
            expect(styleElements.length).toBe(1);
        });
        it("should handle empty styles array", () => {
            const id = "empty-styles";
            const styles = [];
            injectStyles(id, styles);
            const styleElement = document.getElementById(id);
            expect(styleElement).toBeTruthy();
            expect(styleElement?.innerHTML).toBe("");
        });
        it("should handle complex CSS", () => {
            const id = "complex-styles";
            const styles = [
                `
        .container {
          display: flex;
          flex-direction: column;
        }
        .container > .item {
          padding: 1rem;
        }
        @media (min-width: 768px) {
          .container {
            flex-direction: row;
          }
        }
        `,
            ];
            injectStyles(id, styles);
            const styleElement = document.getElementById(id);
            expect(styleElement?.innerHTML).toContain("display: flex");
            expect(styleElement?.innerHTML).toContain("@media");
        });
        it("should handle styles with special characters", () => {
            const id = "special-chars";
            const styles = ['.test::before { content: "\\2022"; }'];
            injectStyles(id, styles);
            const styleElement = document.getElementById(id);
            expect(styleElement?.innerHTML).toContain("::before");
        });
        it("should append to head element", () => {
            const id = "head-test";
            const styles = [".test { color: red; }"];
            const initialChildCount = document.head.childElementCount;
            injectStyles(id, styles);
            expect(document.head.childElementCount).toBe(initialChildCount + 1);
            expect(document.head.contains(document.getElementById(id))).toBe(true);
        });
        it("should handle multiple different style injections", () => {
            injectStyles("styles-1", [".a { color: red; }"]);
            injectStyles("styles-2", [".b { color: blue; }"]);
            injectStyles("styles-3", [".c { color: green; }"]);
            expect(document.getElementById("styles-1")).toBeTruthy();
            expect(document.getElementById("styles-2")).toBeTruthy();
            expect(document.getElementById("styles-3")).toBeTruthy();
            expect(document.querySelectorAll("style").length).toBe(3);
        });
    });
});
