import { describe, it, expect, vi } from "vitest";
import { fireEvent } from "../event";
describe("Event Service", () => {
    describe("fireEvent", () => {
        it("should fire custom event on document", () => {
            const handler = vi.fn();
            document.addEventListener("test-event", handler);
            fireEvent(document, "test-event");
            expect(handler).toHaveBeenCalled();
            document.removeEventListener("test-event", handler);
        });
        it("should fire custom event on element", () => {
            const element = document.createElement("div");
            const handler = vi.fn();
            element.addEventListener("test-event", handler);
            fireEvent(element, "test-event");
            expect(handler).toHaveBeenCalled();
        });
        it("should pass detail object", () => {
            const handler = vi.fn();
            const detail = { foo: "bar", count: 42 };
            document.addEventListener("custom-event", handler);
            fireEvent(document, "custom-event", detail);
            expect(handler).toHaveBeenCalled();
            const event = handler.mock.calls[0][0];
            expect(event.detail).toEqual(detail);
            document.removeEventListener("custom-event", handler);
        });
        it("should create bubbling events", () => {
            const parent = document.createElement("div");
            const child = document.createElement("span");
            parent.appendChild(child);
            const parentHandler = vi.fn();
            parent.addEventListener("bubble-test", parentHandler);
            fireEvent(child, "bubble-test");
            expect(parentHandler).toHaveBeenCalled();
        });
        it("should handle events without detail", () => {
            const handler = vi.fn();
            document.addEventListener("no-detail", handler);
            fireEvent(document, "no-detail");
            expect(handler).toHaveBeenCalled();
            const event = handler.mock.calls[0][0];
            expect(event.detail).toEqual({});
            document.removeEventListener("no-detail", handler);
        });
        it("should handle complex detail objects", () => {
            const handler = vi.fn();
            const detail = {
                nested: {
                    deep: {
                        value: "test",
                    },
                },
                array: [1, 2, 3],
                boolean: true,
            };
            document.addEventListener("complex-event", handler);
            fireEvent(document, "complex-event", detail);
            const event = handler.mock.calls[0][0];
            expect(event.detail).toEqual(detail);
            document.removeEventListener("complex-event", handler);
        });
        it("should create CustomEvent instances", () => {
            const handler = vi.fn();
            document.addEventListener("custom-event-check", handler);
            fireEvent(document, "custom-event-check", { test: true });
            const event = handler.mock.calls[0][0];
            expect(event).toBeInstanceOf(CustomEvent);
            document.removeEventListener("custom-event-check", handler);
        });
        it("should work with different event names", () => {
            const handler1 = vi.fn();
            const handler2 = vi.fn();
            document.addEventListener("event-one", handler1);
            document.addEventListener("event-two", handler2);
            fireEvent(document, "event-one");
            fireEvent(document, "event-two");
            expect(handler1).toHaveBeenCalledTimes(1);
            expect(handler2).toHaveBeenCalledTimes(1);
            document.removeEventListener("event-one", handler1);
            document.removeEventListener("event-two", handler2);
        });
    });
});
