import { describe, it, expect } from "vitest";
import { textContainsEmail, getEmailPresenceHelperContent, } from "../email-in-text";
describe("Email in Text Service", () => {
    describe("textContainsEmail", () => {
        it("should detect email in text", () => {
            const element = document.createElement("div");
            element.textContent = "Contact us at test@example.com";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should detect multiple emails", () => {
            const element = document.createElement("div");
            element.textContent = "Emails: test@example.com and admin@example.com";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should return false when no email present", () => {
            const element = document.createElement("div");
            element.textContent = "This is just plain text without emails";
            expect(textContainsEmail(element)).toBe(false);
        });
        it("should handle empty element", () => {
            const element = document.createElement("div");
            element.textContent = "";
            expect(textContainsEmail(element)).toBe(false);
        });
        it("should handle undefined element", () => {
            expect(textContainsEmail(undefined)).toBe(false);
        });
        it("should detect emails with various TLDs", () => {
            const element = document.createElement("div");
            element.textContent = "test@example.co.uk";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should detect emails with numbers", () => {
            const element = document.createElement("div");
            element.textContent = "user123@example.com";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should detect emails with special characters", () => {
            const element = document.createElement("div");
            element.textContent = "user.name+tag@example.com";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should detect uppercase emails", () => {
            const element = document.createElement("div");
            element.textContent = "USER@EXAMPLE.COM";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should detect emails in mixed case", () => {
            const element = document.createElement("div");
            element.textContent = "User@Example.Com";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should not detect incomplete email addresses", () => {
            const element = document.createElement("div");
            element.textContent = "user@";
            expect(textContainsEmail(element)).toBe(false);
        });
        it("should not detect email-like text without domain", () => {
            const element = document.createElement("div");
            element.textContent = "test@localhost";
            expect(textContainsEmail(element)).toBe(false);
        });
        it("should detect email in nested elements", () => {
            const element = document.createElement("div");
            const span = document.createElement("span");
            span.textContent = "contact: test@example.com";
            element.appendChild(span);
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should handle whitespace around emails", () => {
            const element = document.createElement("div");
            element.textContent = "  test@example.com  ";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should detect email at start of text", () => {
            const element = document.createElement("div");
            element.textContent = "test@example.com is the contact";
            expect(textContainsEmail(element)).toBe(true);
        });
        it("should detect email at end of text", () => {
            const element = document.createElement("div");
            element.textContent = "Contact: test@example.com";
            expect(textContainsEmail(element)).toBe(true);
        });
    });
    describe("getEmailPresenceHelperContent", () => {
        it("should return warning message", () => {
            const message = getEmailPresenceHelperContent();
            expect(message).toBeDefined();
            expect(typeof message).toBe("string");
        });
        it("should return non-empty message", () => {
            const message = getEmailPresenceHelperContent();
            expect(message.length).toBeGreaterThan(0);
        });
    });
});
