import { snackbar } from "./user-interface/common/eden-snackbar.js";
import { fetchFormats, fetchTimestamp } from "./services/fetcher.js";
import { disableBrowserTranslation } from "./services/util.js";
import { closeDetailsElements, cleanupTagsAndPreserveContent, randomizeSvgSpriteCalls, cleanupRandomSvgSpriteCallParams, cleanupOldSvgAttributes, } from "./services/cleanup.js";
import { saveZones } from "./services/api.js";
import { decodeToken } from "./services/token.js";
import { fireEvent } from "./services/event.js";
import { fillTemplates, groupTemplates } from "./services/templates.js";
import { EdenMode, } from "./types.js";
import t from "./i18n/index.js";
import { setCSSTranslations } from "./services/css.js";
export default class Eden {
    constructor() {
        this.players = [];
        this.cursorPosition = 0;
        this.init = () => {
            this.zones = [];
            this.templateDescriptors = {};
            this.user = decodeToken();
            cleanupOldSvgAttributes();
            randomizeSvgSpriteCalls();
            setCSSTranslations();
        };
        this.start = async () => {
            this.getTimestamp();
            this.init();
            this.changeMode(EdenMode.connected);
            await this.register();
            await this.analyse();
            import("./user-interface/icons/eden-icon.js");
            import("./user-interface/common/eden-button.js");
            import("./user-interface/common/eden-checkbox.js");
            import("./user-interface/common/eden-input.js");
            import("./user-interface/common/eden-tag-manager.js");
            disableBrowserTranslation();
            const { default: EdenPageUi } = await import("./user-interface/page/eden-main-ui.js");
            this.pageUi = new EdenPageUi();
            document.body.append(this.pageUi);
        };
        this.getTimestamp = async () => {
            const { timestamp } = await fetchTimestamp();
            this.startedAt = timestamp;
        };
        this.register = async () => {
            const bareTemplates = await fetchFormats();
            const templates = fillTemplates(bareTemplates);
            const { default: FormatDescriptor } = await import("./core/template.js");
            Object.entries(templates).forEach(([templateName, template]) => {
                this.templateDescriptors[templateName] = new FormatDescriptor(template);
            });
            this.templates = templates;
            this.groupedTemplates = groupTemplates(templates);
        };
        this.analyse = async () => {
            const zoneElements = Array.from(document.querySelectorAll("[data-eden-zone]"));
            const { default: Zone } = await import("./core/zone.js");
            const zones = await Promise.all(zoneElements.map(async (zoneElement) => {
                const zone = new Zone(zoneElement, this.templateDescriptors);
                await zone.analyse();
                return zone;
            }));
            this.zones = zones;
        };
        this.addZone = async (zoneElement) => {
            const { default: Zone } = await import("./core/zone.js");
            const zone = new Zone(zoneElement, this.templateDescriptors);
            await zone.analyse();
            this.zones.push(zone);
            return zone;
        };
        this.stop = () => {
            this.desactivate();
            this.pageUi?.remove();
            this.cleanRefs();
            this.init();
        };
        this.cleanRefs = () => {
            this.zones.forEach((zone) => zone.cleanRefs());
        };
        this.activate = () => {
            this.zones.map((zone) => zone.activate());
            this.changeMode(EdenMode.editing);
        };
        this.desactivate = () => {
            this.zones.map((zone) => zone.desactivate());
            this.changeMode(EdenMode.connected);
        };
        this.preSave = () => {
            this.zones.map((zone) => zone.preSave());
        };
        this.save = async () => {
            this.desactivate();
            this.preSave();
            this.cleanMarkup();
            const { lang } = window.EdenConfig;
            const pathname = window.location.pathname;
            const { error } = await saveZones({
                lang,
                path: pathname,
                startedAt: this.startedAt,
            });
            this.postSave(error);
        };
        this.postSave = (error) => {
            if (error) {
                return snackbar(error, "error");
            }
            this.zones.map((zone) => zone.postSave());
            fireEvent(document, "eden-content-saved");
            snackbar(t("snackbars.content-saved"), "success");
            randomizeSvgSpriteCalls();
            setTimeout(() => {
                this.startedAt = Date.now();
            }, 100);
        };
        this.cleanMarkup = () => {
            const selectorsToRemove = [
                "grammarly-extension",
                "lt-highlighter",
                "eden-zone-ui",
                "eden-info-ui",
                "eden-block-ui",
                "[data-dashlanecreated]",
                "mci-extension",
                "editor-squiggler",
                '[class="monica-widget"]',
            ];
            const elementsToRemove = Array.from(document.querySelectorAll(selectorsToRemove.join()));
            elementsToRemove.forEach((el) => el.remove());
            const attributesToRemove = [
                "spellcheck",
                "data-lt-tmp-id",
                "data-gramm",
                "data-eden-helper",
                "data-dashlane-rid",
                "data-form-type",
                "data-kwimpalastatus",
                "data-mci-spellchecked",
                "data-full-weeks-obsolete",
                "data-ms-editor",
            ];
            attributesToRemove.forEach((attribute) => {
                const targets = Array.from(document.querySelectorAll(`[${attribute}]`));
                targets.forEach((target) => target.removeAttribute(attribute));
            });
            const classNamesToRemove = ["undefined"];
            classNamesToRemove.forEach((className) => {
                const targets = Array.from(document.querySelectorAll(`.${className}`));
                targets.forEach((target) => target.classList.remove(className));
            });
            cleanupTagsAndPreserveContent("font[style], tcxspan");
            closeDetailsElements();
            cleanupRandomSvgSpriteCallParams();
        };
        this.getActiveBlock = () => {
            return this.activeBlock;
        };
        this.setActiveBlock = (block) => {
            this.activeBlock = block;
        };
        this.logout = async () => {
            if (window.updateEdenDraftMode) {
                const token = window.sessionStorage.getItem("eden_token");
                if (!token)
                    return;
                await window.updateEdenDraftMode(false, token);
            }
            this.desactivate();
            this.pageUi?.remove();
            window.sessionStorage.removeItem("eden_token");
            window.location.reload();
        };
        this.displayInfo = () => {
            this.zones.map((zone) => zone.displayInfo());
            this.changeMode(EdenMode.info);
        };
        this.hideInfo = () => {
            this.zones.map((zone) => zone.hideInfo());
            this.changeMode(EdenMode.connected);
        };
        this.changeMode = (mode) => {
            window.sessionStorage.removeItem("old-eden-mode");
            if (mode) {
                document.documentElement.setAttribute("data-eden-mode", mode);
                window.sessionStorage.setItem("old-eden-mode", mode);
            }
            if (window.setEdenMode) {
                window.setEdenMode(mode);
            }
        };
    }
}
