import { beforeEach, describe, expect, it, vi } from "vitest";
const mocks = vi.hoisted(() => {
    const snackbarMock = vi.fn();
    const fireEventMock = vi.fn();
    const isLocalhostMock = vi.fn(() => false);
    const buildUuidMock = vi.fn(() => "uuid-1");
    const tMock = vi.fn((key) => key);
    const pasteFromClipboardMock = vi.fn();
    const clipboardErrorMock = vi.fn();
    const createdBlocks = [];
    const blockInitMock = vi.fn();
    const blockActivateMock = vi.fn();
    const blockDesactivateMock = vi.fn();
    const blockPreSaveMock = vi.fn();
    const blockPostSaveMock = vi.fn();
    const blockDisplayInfoMock = vi.fn();
    const blockHideInfoMock = vi.fn();
    const blockShowUiMock = vi.fn();
    const blockCleanRefsMock = vi.fn();
    class BlockMock {
        constructor(element, _template, zone, _code, _isCreation) {
            this.element = element;
            this.zone = zone;
            createdBlocks.push(this);
        }
        async init() {
            blockInitMock();
        }
        activate() {
            blockActivateMock();
        }
        desactivate() {
            blockDesactivateMock();
        }
        preSave() {
            blockPreSaveMock();
        }
        postSave() {
            blockPostSaveMock();
        }
        displayInfo() {
            blockDisplayInfoMock();
        }
        hideInfo() {
            blockHideInfoMock();
        }
        showUi() {
            blockShowUiMock();
        }
        cleanRefs() {
            blockCleanRefsMock();
        }
    }
    class EdenZoneUiMock {
        constructor(zone, actions, templates, maxBlocks) {
            const el = document.createElement("div");
            el.setAttribute("data-testid", "eden-zone-ui");
            el.zone = zone;
            el.actions = actions;
            el.templates = templates;
            el.maxBlocks = maxBlocks;
            return el;
        }
    }
    return {
        snackbarMock,
        fireEventMock,
        isLocalhostMock,
        buildUuidMock,
        tMock,
        pasteFromClipboardMock,
        clipboardErrorMock,
        createdBlocks,
        blockInitMock,
        blockActivateMock,
        blockDesactivateMock,
        blockPreSaveMock,
        blockPostSaveMock,
        blockDisplayInfoMock,
        blockHideInfoMock,
        blockShowUiMock,
        blockCleanRefsMock,
        BlockMock,
        EdenZoneUiMock,
    };
});
vi.mock("../block.js", () => ({
    default: mocks.BlockMock,
    pasteFromClipboard: mocks.pasteFromClipboardMock,
    clipboardError: mocks.clipboardErrorMock,
}));
vi.mock("../../user-interface/common/eden-snackbar.js", () => ({
    snackbar: mocks.snackbarMock,
}));
vi.mock("../../services/event.js", () => ({
    fireEvent: mocks.fireEventMock,
}));
vi.mock("../../services/util.js", () => ({
    isLocalhost: mocks.isLocalhostMock,
}));
vi.mock("../../services/uuid.js", () => ({
    default: mocks.buildUuidMock,
}));
vi.mock("../../i18n/index.js", () => ({
    default: mocks.tMock,
}));
vi.mock("../../user-interface/zone/eden-zone-ui.js", () => ({
    default: mocks.EdenZoneUiMock,
}));
import Zone from "../zone";
describe("Zone", () => {
    beforeEach(() => {
        vi.clearAllMocks();
        mocks.createdBlocks.splice(0, mocks.createdBlocks.length);
        Object.defineProperty(navigator, "clipboard", {
            configurable: true,
            value: {
                writeText: vi.fn().mockResolvedValue(undefined),
            },
        });
        mocks.buildUuidMock.mockReturnValue("uuid-1");
        mocks.isLocalhostMock.mockReturnValue(false);
        mocks.pasteFromClipboardMock.mockImplementation(async (position, element) => {
            element.insertAdjacentHTML(position, `<section data-template="tpl"></section>`);
        });
    });
    const makeZoneElement = (dataset) => {
        const el = document.createElement("div");
        Object.entries(dataset ?? {}).forEach(([key, value]) => {
            el.dataset[key] = value;
        });
        return el;
    };
    const makeDescriptors = () => ({
        tpl: {
            createElement: (templateCode) => {
                const section = document.createElement("section");
                section.setAttribute("data-template", templateCode);
                return section;
            },
        },
        other: {
            createElement: (templateCode) => {
                const section = document.createElement("section");
                section.setAttribute("data-template", templateCode);
                return section;
            },
        },
    });
    it("parses dataset options in constructor", () => {
        const element = makeZoneElement({
            edenDefaultContent: "tpl",
            edenMaxBlocks: "2",
            edenTemplates: "tpl,other",
            edenTagsLabel: "Tags",
            edenAllowedTags: "news,events",
            edenAllowMultipleTags: "true",
            edenFixed: "true",
            edenReadOnly: "true",
            edenUuid: "true",
        });
        const zone = new Zone(element, makeDescriptors());
        expect(zone.defaultContent).toBe("tpl");
        expect(zone.maxBlocks).toBe(2);
        expect(zone.templates).toEqual(["tpl", "other"]);
        expect(zone.tagsLabel).toBe("Tags");
        expect(zone.allowedTags).toBe("news,events");
        expect(zone.allowMultipleTags).toBe(true);
        expect(zone.isFixed).toBe(true);
        expect(zone.isReadOnly).toBe(true);
        expect(zone.withUuid).toBe(true);
        expect(zone.hasChanged).toBe(false);
        expect(zone.defaultContentSpawned).toBe(false);
        expect(zone.blocks).toEqual([]);
    });
    it("analyse creates blocks from existing DOM sections", async () => {
        const element = makeZoneElement();
        element.innerHTML = `
      <section data-template="tpl"></section>
      <section data-template="other"></section>
    `;
        const zone = new Zone(element, makeDescriptors());
        await zone.analyse();
        expect(mocks.createdBlocks).toHaveLength(2);
        expect(mocks.blockInitMock).toHaveBeenCalledTimes(2);
        expect(zone.blocks).toHaveLength(2);
    });
    it("analyse ignores sections without template name or without descriptor", async () => {
        const element = makeZoneElement();
        element.innerHTML = `
      <section></section>
      <section data-template="missing"></section>
      <section data-template="tpl"></section>
    `;
        const zone = new Zone(element, makeDescriptors());
        await zone.analyse();
        expect(mocks.createdBlocks).toHaveLength(1);
        expect(zone.blocks).toHaveLength(1);
    });
    it("analyse spawns default content once when empty", async () => {
        const element = makeZoneElement({ edenDefaultContent: "tpl" });
        const zone = new Zone(element, makeDescriptors());
        await zone.analyse();
        expect(zone.defaultContentSpawned).toBe(true);
        expect(element.querySelectorAll("section[data-template]")).toHaveLength(1);
    });
    it("analyse swallows block.init errors", async () => {
        const element = makeZoneElement();
        element.innerHTML = `<section data-template="tpl"></section>`;
        const zone = new Zone(element, makeDescriptors());
        const initSpy = vi
            .spyOn(mocks.BlockMock.prototype, "init")
            .mockRejectedValueOnce(new Error("boom"));
        await expect(zone.analyse()).resolves.toBeUndefined();
        expect(initSpy).toHaveBeenCalled();
        expect(zone.blocks).toHaveLength(1);
    });
    it("showUi appends EdenZoneUi when empty and not read-only", async () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        zone.blocks = [];
        await zone.showUi();
        expect(element.querySelector('[data-testid="eden-zone-ui"]')).toBeTruthy();
    });
    it("showUi does nothing when read-only or not empty", async () => {
        const element = makeZoneElement({ edenReadOnly: "true" });
        const zone = new Zone(element, makeDescriptors());
        zone.blocks = [];
        await zone.showUi();
        expect(element.querySelector('[data-testid="eden-zone-ui"]')).toBeNull();
        zone.isReadOnly = false;
        zone.blocks = [
            new mocks.BlockMock(document.createElement("section"), {}, zone, "tpl", false),
        ];
        await zone.showUi();
        expect(element.querySelector('[data-testid="eden-zone-ui"]')).toBeNull();
    });
    it("hideUi removes existing UI", async () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        zone.blocks = [];
        await zone.showUi();
        expect(element.querySelector('[data-testid="eden-zone-ui"]')).toBeTruthy();
        zone.hideUi();
        expect(element.querySelector('[data-testid="eden-zone-ui"]')).toBeNull();
    });
    it("activate/desactivate delegate to blocks and UI", async () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        zone.blocks = [
            new mocks.BlockMock(document.createElement("section"), {}, zone, "tpl", false),
        ];
        await zone.activate();
        expect(mocks.blockActivateMock).toHaveBeenCalledTimes(1);
        zone.desactivate();
        expect(mocks.blockDesactivateMock).toHaveBeenCalledTimes(1);
    });
    it("preSave/postSave/displayInfo/hideInfo delegate to blocks", () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        zone.blocks = [
            new mocks.BlockMock(document.createElement("section"), {}, zone, "tpl", false),
            new mocks.BlockMock(document.createElement("section"), {}, zone, "other", false),
        ];
        zone.preSave();
        zone.postSave();
        zone.displayInfo();
        zone.hideInfo();
        expect(mocks.blockPreSaveMock).toHaveBeenCalledTimes(2);
        expect(mocks.blockPostSaveMock).toHaveBeenCalledTimes(2);
        expect(mocks.blockDisplayInfoMock).toHaveBeenCalledTimes(2);
        expect(mocks.blockHideInfoMock).toHaveBeenCalledTimes(2);
    });
    it("addUuids adds uuids when enabled", () => {
        const element = makeZoneElement({ edenUuid: "true" });
        element.innerHTML = `
      <section data-template="tpl"></section>
      <section data-template="other"></section>
    `;
        const zone = new Zone(element, makeDescriptors());
        mocks.buildUuidMock
            .mockReturnValueOnce("uuid-1")
            .mockReturnValueOnce("uuid-2");
        zone.addUuids();
        const sections = element.querySelectorAll("section[data-template]");
        expect(sections[0].getAttribute("data-uuid")).toBe("uuid-1");
        expect(sections[1].getAttribute("data-uuid")).toBe("uuid-2");
    });
    it("addUuids is a no-op when disabled", () => {
        const element = makeZoneElement({ edenUuid: "false" });
        element.innerHTML = `<section data-template="tpl"></section>`;
        const zone = new Zone(element, makeDescriptors());
        zone.addUuids();
        expect(element.querySelector("section")?.hasAttribute("data-uuid")).toBe(false);
    });
    it("paste reads from clipboard, re-analyses, and shows snackbar", async () => {
        const element = makeZoneElement({ edenUuid: "true" });
        const zone = new Zone(element, makeDescriptors());
        const analyseSpy = vi.spyOn(zone, "analyse").mockResolvedValue(undefined);
        const activateSpy = vi.spyOn(zone, "activate").mockImplementation(() => { });
        const desactivateSpy = vi
            .spyOn(zone, "desactivate")
            .mockImplementation(() => { });
        await zone.paste();
        expect(mocks.pasteFromClipboardMock).toHaveBeenCalledTimes(1);
        expect(desactivateSpy).toHaveBeenCalledTimes(1);
        expect(analyseSpy).toHaveBeenCalledTimes(1);
        expect(activateSpy).toHaveBeenCalledTimes(1);
        expect(mocks.snackbarMock).toHaveBeenCalledWith("snackbars.pasted-content", "success");
    });
    it("paste shows error snackbar when clipboard read fails", async () => {
        mocks.pasteFromClipboardMock.mockRejectedValueOnce(new Error("nope"));
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        await zone.paste();
        expect(mocks.snackbarMock).toHaveBeenCalledWith("snackbars.paste-error", "error");
    });
    it("paste calls clipboardError when Clipboard API is missing", async () => {
        Object.defineProperty(navigator, "clipboard", {
            configurable: true,
            value: undefined,
        });
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        await zone.paste();
        expect(mocks.clipboardErrorMock).toHaveBeenCalledTimes(1);
    });
    it("create appends new block, shows UI, and fires event", async () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        await zone.create("tpl", undefined, true);
        expect(element.querySelectorAll("section[data-template]")).toHaveLength(1);
        expect(mocks.blockShowUiMock).toHaveBeenCalledTimes(1);
        expect(mocks.fireEventMock).toHaveBeenCalledWith(element, "eden-block-created", expect.objectContaining({
            block: expect.any(HTMLElement),
        }));
    });
    it("create does not show block UI when showBlockUi=false", async () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        await zone.create("tpl", undefined, false);
        expect(element.querySelectorAll("section[data-template]")).toHaveLength(1);
        expect(mocks.blockShowUiMock).not.toHaveBeenCalled();
    });
    it("createBlock respects maxBlocks and missing template descriptors", async () => {
        const element = makeZoneElement({ edenMaxBlocks: "1" });
        const zone = new Zone(element, makeDescriptors());
        const block1 = await zone.createBlock("tpl");
        expect(block1).toBeTruthy();
        const block2 = await zone.createBlock("other");
        expect(block2).toBeUndefined();
        zone.maxBlocks = undefined;
        const block3 = await zone.createBlock("missing");
        expect(block3).toBeUndefined();
    });
    it("createBlock can skip activation", async () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        await zone.createBlock("tpl", undefined, false);
        expect(mocks.blockActivateMock).not.toHaveBeenCalled();
    });
    it("copy writes zone HTML to clipboard and shows snackbar", async () => {
        const element = makeZoneElement();
        element.innerHTML = `<section data-template="tpl"></section>`;
        const zone = new Zone(element, makeDescriptors());
        await zone.copy();
        const writeText = navigator.clipboard.writeText;
        expect(writeText).toHaveBeenCalledTimes(1);
        expect(String(writeText.mock.calls[0][0])).toContain("fromEden:");
        expect(mocks.snackbarMock).toHaveBeenCalledWith("snackbars.copied-blocks", "success");
    });
    it("copy shows error snackbar on clipboard errors", async () => {
        const writeText = vi.fn().mockRejectedValueOnce(new Error("fail"));
        Object.defineProperty(navigator, "clipboard", {
            configurable: true,
            value: { writeText },
        });
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        await zone.copy();
        expect(mocks.snackbarMock).toHaveBeenCalledWith("snackbars.copy-error", "error");
    });
    it("copy calls clipboardError when Clipboard API is missing", async () => {
        Object.defineProperty(navigator, "clipboard", {
            configurable: true,
            value: undefined,
        });
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        await zone.copy();
        expect(mocks.clipboardErrorMock).toHaveBeenCalledTimes(1);
    });
    it("cleanRefs cleans blocks and deletes refs", () => {
        const element = makeZoneElement();
        const zone = new Zone(element, makeDescriptors());
        zone.blocks = [
            new mocks.BlockMock(document.createElement("section"), {}, zone, "tpl", false),
        ];
        zone.cleanRefs();
        expect(mocks.blockCleanRefsMock).toHaveBeenCalledTimes(1);
        expect(zone.blocks).toBeUndefined();
        expect(zone.templateDescriptors).toBeUndefined();
    });
    it("canAddBlock returns true when blocks undefined", () => {
        const element = makeZoneElement({ edenMaxBlocks: "1" });
        const zone = new Zone(element, makeDescriptors());
        zone.blocks = undefined;
        expect(zone.canAddBlock()).toBe(true);
    });
});
