import { describe, it, expect } from "vitest";
import Template from "../template";
describe("Template", () => {
    describe("constructor", () => {
        it("should initialize from template definition", () => {
            const definition = {
                name: "TestTemplate",
                markup: { container: { ".title": "<h1></h1>" } },
                cssClass: "test-class",
                group: "Textes",
            };
            const template = new Template(definition);
            expect(template.name).toBe("TestTemplate");
            expect(template.templates).toEqual(definition.markup);
            expect(template.cssClass).toBe("test-class");
            expect(template.group).toBe("Textes");
        });
        it("should initialize from another Template instance", () => {
            const original = new Template({
                name: "Original",
                markup: { container: {} },
            });
            const copy = new Template(original);
            expect(copy.name).toBe("Original");
            expect(copy.templates).toEqual(original.templates);
        });
        it("should handle variants", () => {
            const template = new Template({
                name: "WithVariants",
                markup: {},
                variants: [
                    { name: "Primary", cssClass: "primary" },
                    { name: "Secondary", cssClass: "secondary" },
                ],
                defaultVariantName: "Primary",
            });
            expect(template.variants).toHaveLength(2);
            expect(template.defaultVariantName).toBe("Primary");
        });
        it("should handle attributes", () => {
            const template = new Template({
                name: "WithAttributes",
                markup: {},
                attributes: [
                    {
                        property: "data-id",
                        name: "ID",
                        placeholder: "Enter ID",
                    },
                ],
            });
            expect(template.attributes).toHaveLength(1);
            expect(template.attributes?.[0].property).toBe("data-id");
        });
        it("should handle groups", () => {
            const template = new Template({
                name: "WithGroups",
                markup: {},
                groups: ["Textes", "Images"],
            });
            expect(template.groups).toEqual(["Textes", "Images"]);
        });
    });
    describe("createElement", () => {
        it("should create a basic section element", () => {
            const template = new Template({
                name: "Basic",
                markup: {},
            });
            const element = template.createElement("basic");
            expect(element.tagName).toBe("SECTION");
            expect(element.getAttribute("data-template")).toBe("basic");
        });
        it("should add CSS classes", () => {
            const template = new Template({
                name: "Styled",
                markup: {},
                cssClass: "container fluid",
            });
            const element = template.createElement("styled");
            expect(element.classList.contains("container")).toBe(true);
            expect(element.classList.contains("fluid")).toBe(true);
        });
        it("should add variant class if provided", () => {
            const template = new Template({
                name: "WithVariant",
                markup: {},
                variants: [{ name: "Dark", cssClass: "dark" }],
            });
            const element = template.createElement("with-variant", undefined, "dark");
            expect(element.classList.contains("dark")).toBe(true);
        });
        it("should add UUID when zone requires it", () => {
            const template = new Template({
                name: "WithUuid",
                markup: {},
            });
            const mockZone = { withUuid: true };
            const element = template.createElement("with-uuid", mockZone);
            expect(element.hasAttribute("data-uuid")).toBe(true);
            expect(element.getAttribute("data-uuid")).toMatch(/^eden-[0-9a-f-]+$/i);
        });
        it("should not add UUID when zone doesn't require it", () => {
            const template = new Template({
                name: "WithoutUuid",
                markup: {},
            });
            const mockZone = { withUuid: false };
            const element = template.createElement("without-uuid", mockZone);
            expect(element.hasAttribute("data-uuid")).toBe(false);
        });
        it("should handle empty CSS class", () => {
            const template = new Template({
                name: "NoClass",
                markup: {},
                cssClass: "   ",
            });
            const element = template.createElement("no-class");
            expect(element.className).toBe("");
        });
    });
    describe("getVariant", () => {
        it("should return variant CSS class if found", () => {
            const template = new Template({
                name: "Variants",
                markup: {},
                variants: [
                    { name: "Primary", cssClass: "primary" },
                    { name: "Secondary", cssClass: "secondary" },
                ],
            });
            const element = document.createElement("section");
            element.classList.add("primary");
            const variant = template.getVariant(element);
            expect(variant).toBe("primary");
        });
        it("should return undefined if no variant found", () => {
            const template = new Template({
                name: "Variants",
                markup: {},
                variants: [
                    { name: "Primary", cssClass: "primary" },
                    { name: "Secondary", cssClass: "secondary" },
                ],
            });
            const element = document.createElement("section");
            element.classList.add("other-class");
            const variant = template.getVariant(element);
            expect(variant).toBeUndefined();
        });
        it("should return undefined if template has no variants", () => {
            const template = new Template({
                name: "NoVariants",
                markup: {},
            });
            const element = document.createElement("section");
            const variant = template.getVariant(element);
            expect(variant).toBeUndefined();
        });
    });
    describe("getAttributes", () => {
        it("should extract attributes from element", () => {
            const template = new Template({
                name: "WithAttrs",
                markup: {},
                attributes: [
                    {
                        property: "data-id",
                        name: "ID",
                        placeholder: "Enter ID",
                    },
                    {
                        property: "data-title",
                        name: "Title",
                    },
                ],
            });
            const element = document.createElement("section");
            element.setAttribute("data-id", "123");
            element.setAttribute("data-title", "Test Title");
            const attributes = template.getAttributes(element);
            expect(attributes).toEqual({
                "data-id": {
                    name: "ID",
                    value: "123",
                    placeholder: "Enter ID",
                    options: undefined,
                    allowMultipleOptions: undefined,
                    helper: undefined,
                },
                "data-title": {
                    name: "Title",
                    value: "Test Title",
                    placeholder: "",
                    options: undefined,
                    allowMultipleOptions: undefined,
                    helper: undefined,
                },
            });
        });
        it("should handle missing attributes", () => {
            const template = new Template({
                name: "WithAttrs",
                markup: {},
                attributes: [
                    {
                        property: "data-id",
                        name: "ID",
                    },
                ],
            });
            const element = document.createElement("section");
            const attributes = template.getAttributes(element);
            expect(attributes).toEqual({
                "data-id": {
                    name: "ID",
                    value: null,
                    placeholder: "",
                    options: undefined,
                    allowMultipleOptions: undefined,
                    helper: undefined,
                },
            });
        });
        it("should return undefined if template has no attributes", () => {
            const template = new Template({
                name: "NoAttrs",
                markup: {},
            });
            const element = document.createElement("section");
            const attributes = template.getAttributes(element);
            expect(attributes).toBeUndefined();
        });
        it("should return undefined if attributes array is empty", () => {
            const template = new Template({
                name: "EmptyAttrs",
                markup: {},
                attributes: [],
            });
            const element = document.createElement("section");
            const attributes = template.getAttributes(element);
            expect(attributes).toBeUndefined();
        });
        it("should handle attributes with options", () => {
            const template = new Template({
                name: "WithOptions",
                markup: {},
                attributes: [
                    {
                        property: "data-color",
                        name: "Color",
                        options: ["red", "blue"],
                        allowMultipleOptions: true,
                    },
                ],
            });
            const element = document.createElement("section");
            element.setAttribute("data-color", "red");
            const attributes = template.getAttributes(element);
            expect(attributes["data-color"]).toMatchObject({
                name: "Color",
                value: "red",
                options: expect.any(Array),
                allowMultipleOptions: true,
            });
        });
        it("should handle attributes with helper text", () => {
            const template = new Template({
                name: "WithHelper",
                markup: {},
                attributes: [
                    {
                        property: "data-url",
                        name: "URL",
                        helper: "Enter a valid URL",
                    },
                ],
            });
            const element = document.createElement("section");
            const attributes = template.getAttributes(element);
            expect(attributes["data-url"]).toBeDefined();
            expect(attributes["data-url"].helper).toBe("Enter a valid URL");
        });
    });
});
