import { beforeEach, describe, expect, it, vi } from "vitest";
vi.mock("../../user-interface/common/eden-snackbar.js", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../../user-interface/common/eden-modal.js", () => ({
    default: class EdenModal {
        constructor(opts) {
            const el = document.createElement("div");
            el.opts = opts;
            return el;
        }
    },
}));
vi.mock("../../services/util.js", () => ({
    isLocalhost: vi.fn(() => false),
    smoothScrollTo: vi.fn(),
}));
vi.mock("../../services/type-of.js", () => ({
    default: (value) => {
        if (value === undefined)
            return "undefined";
        if (value === null)
            return "null";
        if (Array.isArray(value))
            return "array";
        return typeof value;
    },
}));
vi.mock("../../services/event.js", () => ({
    fireEvent: vi.fn(),
}));
const hoisted = vi.hoisted(() => {
    let uuidCounter = 0;
    const nextUuid = () => {
        uuidCounter += 1;
        return `uuid-${uuidCounter}`;
    };
    const resetUuids = () => {
        uuidCounter = 0;
    };
    const makeBrickMock = (name) => { var _a; return _a = class Brick {
            constructor(args) {
                this.init = vi.fn(async () => undefined);
                this.actions = [[{ id: `${name}-action` }]];
                this.watchSelectionChange = false;
                this.args = args;
            }
        },
        _a.brickName = name,
        _a; };
    return { nextUuid, resetUuids, makeBrickMock };
});
vi.mock("../../services/uuid.js", () => ({
    default: hoisted.nextUuid,
}));
vi.mock("../../i18n/index.js", () => ({
    default: (key) => key,
}));
vi.mock("./template.js", () => {
    class MockTemplate {
        constructor(input) {
            if (input instanceof MockTemplate) {
                this.templates = input.templates;
            }
            else {
                this.templates = input?.markup ?? {};
            }
            this.name = input?.name;
            this.cssClass = input?.cssClass;
            this.group = input?.group;
            this.groups = input?.groups;
            this.variants = input?.variants;
            this.defaultVariantName = input?.defaultVariantName;
            this.attributes = input?.attributes;
        }
        getVariant(block) {
            return this.variants
                ? this.variants.find((variant) => block.classList.contains(variant.cssClass))?.cssClass
                : undefined;
        }
        getAttributes(block) {
            return this.attributes && this.attributes?.length > 0
                ? this.attributes.map((attr) => ({
                    property: attr.property || "data-x",
                    value: "1",
                }))
                : [];
        }
    }
    return { default: MockTemplate };
});
vi.mock("../../bricks/Text.js", () => ({
    default: hoisted.makeBrickMock("Text"),
}));
vi.mock("../../bricks/RichText.js", () => ({
    default: hoisted.makeBrickMock("RichText"),
}));
vi.mock("../../bricks/Picture.js", () => ({
    default: hoisted.makeBrickMock("Picture"),
}));
vi.mock("../../bricks/Link.js", () => ({
    default: hoisted.makeBrickMock("Link"),
}));
vi.mock("../../bricks/Icon.js", () => ({
    default: hoisted.makeBrickMock("Icon"),
}));
vi.mock("../../bricks/Slideshow.js", () => ({
    default: hoisted.makeBrickMock("Slideshow"),
}));
vi.mock("../../bricks/Table.js", () => ({
    default: hoisted.makeBrickMock("Table"),
}));
vi.mock("../../bricks/Recursive.js", () => ({
    default: hoisted.makeBrickMock("Recursive"),
}));
vi.mock("../../bricks/Video.js", () => ({
    default: hoisted.makeBrickMock("Video"),
}));
vi.mock("../../bricks/Iframe.js", () => ({
    default: hoisted.makeBrickMock("Iframe"),
}));
vi.mock("../../bricks/GoogleMaps.js", () => ({
    default: hoisted.makeBrickMock("GoogleMaps"),
}));
vi.mock("../../bricks/YoutubePlaylist.js", () => ({
    default: hoisted.makeBrickMock("YoutubePlaylist"),
}));
vi.mock("../../bricks/Spotify.js", () => ({
    default: hoisted.makeBrickMock("Spotify"),
}));
vi.mock("../../user-interface/block/eden-block-ui.js", () => ({
    default: class EdenBlockUi {
        constructor(_block, _actions, _disabled, _templates, _maxBlocks, _obsolete, variant) {
            this.addToolbars = vi.fn();
            this.removeActions = vi.fn();
            const el = document.createElement("div");
            el.setAttribute("data-ui", "eden-block-ui");
            this.element = el;
            this.variant = variant;
            return el;
        }
    },
}));
vi.mock("../../user-interface/block/eden-info-ui.js", () => ({
    default: class EdenInfoUi {
        constructor(_el, _dates, _hidden) {
            return document.createElement("div");
        }
    },
}));
vi.mock("../../user-interface/common/eden-helper.js", () => ({
    default: class EdenHelper {
        constructor(_block) {
            const el = document.createElement("div");
            el.style.width = "100px";
            return el;
        }
    },
}));
vi.mock("../../user-interface/plugin/select-dates-modal.js", () => ({
    default: class SelectDatesModal {
        constructor(_cb, _copied, _selected) { }
        init() {
            return undefined;
        }
    },
}));
vi.mock("../../user-interface/plugin/settings-modal.js", () => ({
    default: class SettingsModal {
        constructor(_cb, _data, _variants, _defaultName) { }
        init() {
            return undefined;
        }
    },
}));
import Block, { clipboardError, pasteFromClipboard } from "../block";
import { smoothScrollTo } from "../../services/util.js";
import { fireEvent } from "../../services/event.js";
import { snackbar } from "../../user-interface/common/eden-snackbar.js";
function mockClipboard() {
    const api = {
        readText: vi.fn(async () => ""),
        writeText: vi.fn(async (_v) => undefined),
    };
    Object.defineProperty(navigator, "clipboard", {
        configurable: true,
        value: api,
    });
    return api;
}
function createZone(element) {
    const zoneEl = element ?? document.createElement("div");
    const zone = {
        element: zoneEl,
        isFixed: false,
        withUuid: true,
        templates: ["Text"],
        maxBlocks: 99,
        tagsLabel: "tags",
        allowedTags: [],
        allowMultipleTags: false,
        blocks: [],
        canAddBlock: vi.fn(() => true),
        showUi: vi.fn(),
        hideUi: vi.fn(),
        activate: vi.fn(),
        desactivate: vi.fn(),
        analyse: vi.fn(async () => undefined),
        createBlock: vi.fn(async (code, _variant) => {
            const el = document.createElement("section");
            el.dataset.template = code;
            const block = new Block(el, { markup: {} }, zone, code, true);
            block.init();
            return block;
        }),
    };
    return zone;
}
beforeEach(() => {
    vi.clearAllMocks();
    hoisted.resetUuids();
    document.body.innerHTML = "";
    window.EdenConfig = {
        apiPath: "/api/eden",
        cdnUrl: "https://cdn.example.com",
        allowClearTextEmail: false,
        locale: "en",
        timeProgrammableBlocks: true,
    };
    let activeBlock = undefined;
    window.Eden = {
        zones: [],
        getActiveBlock: () => activeBlock,
        setActiveBlock: (b) => {
            activeBlock = b;
        },
    };
    document.execCommand = vi.fn();
});
class EdenModalElement extends HTMLElement {
}
class EdenSnackbarElement extends HTMLElement {
}
class EdenBlockUiElement extends HTMLElement {
}
if (!customElements.get("eden-modal")) {
    customElements.define("eden-modal", EdenModalElement);
}
if (!customElements.get("eden-snackbar")) {
    customElements.define("eden-snackbar", EdenSnackbarElement);
}
if (!customElements.get("eden-block-ui")) {
    customElements.define("eden-block-ui", EdenBlockUiElement);
}
describe("pasteFromClipboard", () => {
    it("inserts fromEden content and scrolls to inserted element", async () => {
        const clipboard = mockClipboard();
        clipboard.readText.mockResolvedValueOnce('fromEden:<section id="x"></section>');
        const host = document.createElement("div");
        host.innerHTML = '<section id="ref"></section>';
        const ref = host.querySelector("#ref");
        await pasteFromClipboard("beforebegin", ref);
        expect(host.querySelector("#x")).toBeTruthy();
        expect(smoothScrollTo).toHaveBeenCalledTimes(1);
    });
    it("does nothing when clipboard is not fromEden, still scrolls", async () => {
        const clipboard = mockClipboard();
        clipboard.readText.mockResolvedValueOnce("hello");
        const host = document.createElement("div");
        host.innerHTML =
            '<section id="ref"></section><section id="after"></section>';
        const ref = host.querySelector("#ref");
        await pasteFromClipboard("afterend", ref);
        expect(host.querySelectorAll("section").length).toBe(2);
        expect(smoothScrollTo).toHaveBeenCalledTimes(1);
    });
});
describe("clipboardError", () => {
    it("appends an EdenModal element to body", () => {
        expect(document.body.children.length).toBe(0);
        clipboardError();
        expect(document.body.children.length).toBe(1);
        const appended = document.body.firstElementChild;
        expect(appended.opts.title).toBe("snackbars.config-error-title");
        expect(appended.opts.content).toBe("snackbars.clipboard-config-error");
    });
});
describe("Block", () => {
    it("init() wires listeners and recursively instantiates bricks", async () => {
        const zone = createZone();
        const el = document.createElement("section");
        el.innerHTML = `<div class="a"><span class="b"></span></div>`;
        const template = {
            markup: {
                ".a .b": {
                    type: "Recursive",
                    options: { ok: true },
                    markup: {
                        ".c": { type: "Text", options: {}, markup: {} },
                    },
                },
            },
        };
        const block = new Block(el, template, zone, "Text", true);
        const addSpy = vi.spyOn(el, "addEventListener");
        await block.init();
        expect(addSpy).toHaveBeenCalledWith("mouseenter", expect.any(Function));
        expect(addSpy).toHaveBeenCalledWith("mouseleave", expect.any(Function));
        expect(block.instances?.length).toBeGreaterThan(0);
    });
    it("handlePaste() cleans content and inserts HTML", async () => {
        const zone = createZone();
        const el = document.createElement("section");
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        const ev = {
            stopPropagation: vi.fn(),
            preventDefault: vi.fn(),
            clipboardData: {
                getData: vi.fn(() => "a&nbsp;b\nline2"),
            },
        };
        await block.handlePaste(ev);
        expect(ev.stopPropagation).toHaveBeenCalled();
        expect(ev.preventDefault).toHaveBeenCalled();
        expect(document.execCommand).toHaveBeenCalledWith("insertHTML", false, "ab<br />line2");
    });
    it("checkDates() sets datesActive when currently active", () => {
        const zone = createZone();
        const el = document.createElement("section");
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        const now = Date.now();
        const start = new Date(now - 60000).toISOString();
        const end = new Date(now + 60000).toISOString();
        el.dataset.dates = JSON.stringify([[start, end]]);
        block.checkDates();
        expect(el.dataset.datesActive).toBeTruthy();
        expect(el.dataset.datesScheduled).toBeUndefined();
    });
    it("checkDates() sets datesScheduled when scheduled in the future", () => {
        const zone = createZone();
        const el = document.createElement("section");
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        const now = Date.now();
        const start = new Date(now + 60000).toISOString();
        const end = new Date(now + 120000).toISOString();
        el.dataset.dates = JSON.stringify([[start, end]]);
        block.checkDates();
        expect(el.dataset.datesActive).toBeUndefined();
        expect(el.dataset.datesScheduled).toBeTruthy();
    });
    it("cleanupDatesCheck() clears datesActive and datesScheduled", () => {
        const zone = createZone();
        const el = document.createElement("section");
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        el.dataset.datesActive = "x";
        el.dataset.datesScheduled = "y";
        block.cleanupDatesCheck();
        expect(el.dataset.datesActive).toBeUndefined();
        expect(el.dataset.datesScheduled).toBeUndefined();
    });
    it("setOrUpdateUuids() assigns new uuids for duplicates/missing", () => {
        const zoneEl = document.createElement("div");
        zoneEl.innerHTML = `
      <section data-template="Text" data-uuid="dup"></section>
      <section data-template="Text" data-uuid="dup"></section>
      <section data-template="Text"></section>
    `;
        const zone = createZone(zoneEl);
        const el = zoneEl.querySelector("section");
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        block.setOrUpdateUuids();
        const blocks = Array.from(zoneEl.querySelectorAll("section"));
        const uuids = blocks.map((b) => b.dataset.uuid);
        expect(uuids.filter(Boolean).length).toBe(3);
        expect(new Set(uuids).size).toBe(3);
    });
    it("changeSettings() updates id/tags/hidden/attributes + variant", () => {
        const zone = createZone();
        const el = document.createElement("section");
        el.classList.add("old");
        expect(el.classList.contains("old")).toBe(true);
        const block = new Block(el, {
            markup: {},
            variants: [{ cssClass: "old" }, { cssClass: "new" }],
        }, zone, "Text", true);
        expect(block.template).toBeDefined();
        block.ui = { variant: undefined };
        block.changeSettings({
            id: "my-id",
            tags: "a,b",
            hidden: true,
            variant: "new",
            attributes: [
                { property: "data-p", value: "1" },
                { property: "data-remove", value: "" },
            ],
        });
        expect(el.id).toBe("my-id");
        expect(el.dataset.tags).toBe("a,b");
        expect(el.dataset.hidden).toBe("true");
        expect(el.hasAttribute("data-nosnippet")).toBe(true);
        expect(el.getAttribute("data-p")).toBe("1");
        expect(el.hasAttribute("data-remove")).toBe(false);
        expect(el.classList.contains("new")).toBe(true);
    });
    it("remove() removes element, updates zone.blocks, fires event", () => {
        const zoneEl = document.createElement("div");
        const zone = createZone(zoneEl);
        const el = document.createElement("section");
        el.dataset.template = "Text";
        zoneEl.appendChild(el);
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        zone.blocks = [block];
        block.remove();
        expect(fireEvent).toHaveBeenCalledWith(zoneEl, "eden-block-removed", {
            block: el,
        });
        expect(zone.blocks.length).toBe(0);
        expect(zone.showUi).toHaveBeenCalled();
    });
    it("copy()/manualCopy writes fromEden content and shows snackbar", async () => {
        const zoneEl = document.createElement("div");
        const zone = createZone(zoneEl);
        const el = document.createElement("section");
        el.dataset.template = "Text";
        zoneEl.appendChild(el);
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        const clipboard = mockClipboard();
        await block.copyBlock();
        expect(clipboard.writeText).toHaveBeenCalled();
        expect(snackbar).toHaveBeenCalled();
    });
    it("showUi()/hideUi patches <a><details> toggle handler", async () => {
        const zoneEl = document.createElement("div");
        const zone = createZone(zoneEl);
        const el = document.createElement("section");
        el.dataset.template = "Text";
        el.innerHTML = `<a href="#"><details><summary>Hi</summary><div>Body</div></details></a>`;
        zoneEl.appendChild(el);
        const block = new Block(el, { markup: {} }, zone, "Text", true);
        window.Eden.zones = [zone];
        await block.showUi(new MouseEvent("click"));
        const details = el.querySelector("details");
        expect(details.hasAttribute("open")).toBe(false);
        details.click();
        expect(details.hasAttribute("open")).toBe(true);
        details.click();
        expect(details.hasAttribute("open")).toBe(false);
        block.hideUi();
    });
});
