import { describe, it, expect, beforeEach, vi, afterEach } from "vitest";
import YoutubePlaylist from "../YoutubePlaylist";
vi.mock("../../services/url", () => ({
    parseUrl: vi.fn(),
}));
vi.mock("../../user-interface/plugin/youtube-playlist-modal.js", () => ({
    default: vi.fn(),
}));
vi.mock("../../user-interface/common/eden-snackbar.js", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../../i18n/index.js", () => ({
    default: vi.fn((key) => key),
}));
import { parseUrl } from "../../services/url";
import { snackbar } from "../../user-interface/common/eden-snackbar.js";
import YoutubePlaylistModal from "../../user-interface/plugin/youtube-playlist-modal.js";
describe("YoutubePlaylist Brick", () => {
    let mockBlock;
    let mockElement;
    let mockParentBricks;
    let constructorParams;
    const createMockParseUrl = (url) => {
        if (!url.startsWith("http"))
            return undefined;
        const parser = document.createElement("a");
        parser.href = url;
        return {
            url,
            protocol: parser.protocol,
            host: parser.host,
            hostname: parser.hostname,
            port: parser.port,
            path: parser.pathname,
            search: parser.search,
            hash: parser.hash,
            searchObject: {},
        };
    };
    beforeEach(() => {
        vi.clearAllMocks();
        mockBlock = {
            element: document.createElement("div"),
            obsolete: false,
            manageActions: vi.fn(),
        };
        mockElement = document.createElement("div");
        mockParentBricks = [];
        constructorParams = {
            block: mockBlock,
            element: mockElement,
            parentBricks: mockParentBricks,
        };
    });
    afterEach(() => {
        document.body.innerHTML = "";
    });
    describe("Constructor", () => {
        it("should initialize with correct properties", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            expect(youtubePlaylist.element).toBe(mockElement);
            expect(youtubePlaylist.block).toBe(mockBlock);
            expect(youtubePlaylist.parentBricks).toBe(mockParentBricks);
        });
        it("should create mask element with correct styles", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            expect(youtubePlaylist["mask"]).toBeInstanceOf(HTMLDivElement);
            expect(youtubePlaylist["mask"].style.position).toBe("absolute");
            expect(youtubePlaylist["mask"].style.top).toBe("0px");
            expect(youtubePlaylist["mask"].style.bottom).toBe("0px");
            expect(youtubePlaylist["mask"].style.left).toBe("0px");
            expect(youtubePlaylist["mask"].style.right).toBe("0px");
        });
        it("should initialize with actions array containing playlist action", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            expect(youtubePlaylist.actions).toBeDefined();
            expect(youtubePlaylist.actions).toHaveLength(1);
            const action = youtubePlaylist.actions[0];
            expect(action).toHaveProperty("tooltip", "bricks.youtube-playlist.add-playlist");
            expect(action).toHaveProperty("icon", "movie");
            expect(action).toHaveProperty("action");
            expect(typeof action.action).toBe("function");
        });
        it("should store parent bricks", () => {
            const parentBricks = [
                new YoutubePlaylist({
                    block: mockBlock,
                    element: mockElement,
                    parentBricks: [],
                }),
            ];
            const youtubePlaylist = new YoutubePlaylist({
                block: mockBlock,
                element: mockElement,
                parentBricks,
            });
            expect(youtubePlaylist.parentBricks).toBe(parentBricks);
        });
    });
    describe("Actions", () => {
        it("should open YoutubePlaylistModal when action is triggered without existing iframe", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const action = youtubePlaylist.actions[0].action;
            action();
            expect(YoutubePlaylistModal).toHaveBeenCalledWith(youtubePlaylist.updateUrl, "");
        });
        it("should open YoutubePlaylistModal with existing iframe data", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://www.youtube.com/embed/videoseries?list=PLxyz123");
            mockElement.appendChild(iframe);
            const action = youtubePlaylist.actions[0].action;
            action();
            expect(YoutubePlaylistModal).toHaveBeenCalledWith(youtubePlaylist.updateUrl, "https://www.youtube.com/embed/videoseries?list=PLxyz123");
        });
        it("should use default empty string if iframe has no src attribute", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const action = youtubePlaylist.actions[0].action;
            action();
            expect(YoutubePlaylistModal).toHaveBeenCalledWith(youtubePlaylist.updateUrl, "");
        });
    });
    describe("updateUrl", () => {
        beforeEach(() => {
            vi.mocked(parseUrl).mockImplementation(createMockParseUrl);
        });
        it("should update existing iframe with valid YouTube URL", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const validUrl = "https://www.youtube.com/playlist?list=PLxyz123";
            vi.mocked(parseUrl).mockReturnValue({
                url: validUrl,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLxyz123",
                hash: "",
                searchObject: { list: "PLxyz123" },
            });
            youtubePlaylist.updateUrl(validUrl);
            expect(iframe.getAttribute("src")).toBe(validUrl);
        });
        it("should update existing iframe with valid YouTube nocookie URL", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const validUrl = "https://www.youtube-nocookie.com/embed/videoseries?list=PLabc456";
            vi.mocked(parseUrl).mockReturnValue({
                url: validUrl,
                protocol: "https:",
                host: "www.youtube-nocookie.com",
                hostname: "www.youtube-nocookie.com",
                port: "",
                path: "/embed/videoseries",
                search: "?list=PLabc456",
                hash: "",
                searchObject: { list: "PLabc456" },
            });
            youtubePlaylist.updateUrl(validUrl);
            expect(iframe.getAttribute("src")).toBe(validUrl);
        });
        it("should create new iframe if none exists with YouTube URL", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const newUrl = "https://www.youtube.com/playlist?list=PLnew789";
            vi.mocked(parseUrl).mockReturnValue({
                url: newUrl,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLnew789",
                hash: "",
                searchObject: { list: "PLnew789" },
            });
            youtubePlaylist.updateUrl(newUrl);
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).not.toBeNull();
            expect(iframe.getAttribute("src")).toBe(newUrl);
            expect(iframe.getAttribute("frameborder")).toBe("0");
            expect(iframe.hasAttribute("webkitAllowFullScreen")).toBe(true);
            expect(iframe.hasAttribute("mozallowfullscreen")).toBe(true);
            expect(iframe.hasAttribute("allowfullscreen")).toBe(true);
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(true);
        });
        it("should show error snackbar for unsupported video service", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue({
                url: "https://vimeo.com/123456",
                protocol: "https:",
                host: "vimeo.com",
                hostname: "vimeo.com",
                port: "",
                path: "/123456",
                search: "",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("https://vimeo.com/123456");
            expect(snackbar).toHaveBeenCalledWith("snackbars.video-service-not-supported", "error");
        });
        it("should show error snackbar for invalid URL (no host)", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue({
                url: "not-a-url",
                protocol: "",
                host: "",
                hostname: "",
                port: "",
                path: "",
                search: "",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("not-a-url");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
        it("should show error snackbar for empty URL", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue(undefined);
            youtubePlaylist.updateUrl("");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
        it("should show error snackbar when parseUrl returns undefined", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue(undefined);
            youtubePlaylist.updateUrl("invalid");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
        it("should show error snackbar when parseUrl returns object without host", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue({
                url: "something",
                protocol: "",
                host: "",
                hostname: "",
                port: "",
                path: "",
                search: "",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("something");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
        it("should handle unsupported host gracefully", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://www.youtube.com/old");
            mockElement.appendChild(iframe);
            vi.mocked(parseUrl).mockReturnValue({
                url: "https://www.dailymotion.com/video/xyz",
                protocol: "https:",
                host: "www.dailymotion.com",
                hostname: "www.dailymotion.com",
                port: "",
                path: "/video/xyz",
                search: "",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("https://www.dailymotion.com/video/xyz");
            expect(iframe.getAttribute("src")).toBe("https://www.youtube.com/old");
            expect(snackbar).toHaveBeenCalledWith("snackbars.video-service-not-supported", "error");
        });
    });
    describe("activate", () => {
        it("should add click event listener to element", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const addEventListenerSpy = vi.spyOn(mockElement, "addEventListener");
            youtubePlaylist.activate();
            expect(addEventListenerSpy).toHaveBeenCalledWith("click", youtubePlaylist.handleClick);
        });
        it("should append mask if iframe exists", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            youtubePlaylist.activate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(true);
        });
        it("should not append mask if no iframe exists", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            youtubePlaylist.activate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(false);
        });
        it("should do nothing if element is null", () => {
            const nullElementParams = {
                ...constructorParams,
                element: null,
            };
            const youtubePlaylist = new YoutubePlaylist(nullElementParams);
            expect(() => youtubePlaylist.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener from element", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const removeEventListenerSpy = vi.spyOn(mockElement, "removeEventListener");
            youtubePlaylist.desactivate();
            expect(removeEventListenerSpy).toHaveBeenCalledWith("click", youtubePlaylist.handleClick);
        });
        it("should remove mask from DOM", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            mockElement.appendChild(youtubePlaylist["mask"]);
            youtubePlaylist.desactivate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(false);
        });
        it("should not throw if mask is not in DOM", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            expect(() => youtubePlaylist.desactivate()).not.toThrow();
        });
        it("should do nothing if element is null", () => {
            const nullElementParams = {
                ...constructorParams,
                element: null,
            };
            const youtubePlaylist = new YoutubePlaylist(nullElementParams);
            expect(() => youtubePlaylist.desactivate()).not.toThrow();
        });
    });
    describe("Integration: activate and desactivate cycle", () => {
        it("should properly add and remove mask during activate/desactivate cycle", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            youtubePlaylist.activate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(true);
            youtubePlaylist.desactivate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(false);
            youtubePlaylist.activate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(true);
        });
    });
    describe("handleClick", () => {
        it("should call manageActions when clicked", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const mockEvent = new MouseEvent("click");
            const preventDefaultSpy = vi.spyOn(mockEvent, "preventDefault");
            const stopPropagationSpy = vi.spyOn(mockEvent, "stopPropagation");
            youtubePlaylist.handleClick(mockEvent);
            expect(preventDefaultSpy).toHaveBeenCalled();
            expect(stopPropagationSpy).toHaveBeenCalled();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([youtubePlaylist]);
        });
        it("should include parent bricks in manageActions call", () => {
            const parentBrick = new YoutubePlaylist({
                block: mockBlock,
                element: document.createElement("div"),
                parentBricks: [],
            });
            const paramsWithParent = {
                ...constructorParams,
                parentBricks: [parentBrick],
            };
            const youtubePlaylist = new YoutubePlaylist(paramsWithParent);
            const mockEvent = new MouseEvent("click");
            youtubePlaylist.handleClick(mockEvent);
            expect(mockBlock.manageActions).toHaveBeenCalledWith([
                parentBrick,
                youtubePlaylist,
            ]);
        });
        it("should work without event parameter", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            expect(() => youtubePlaylist.handleClick()).not.toThrow();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([youtubePlaylist]);
        });
    });
    describe("Edge cases and error handling", () => {
        beforeEach(() => {
            vi.mocked(parseUrl).mockImplementation(createMockParseUrl);
        });
        it("should handle updateUrl with iframe but invalid url", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://www.youtube.com/embed/old");
            mockElement.appendChild(iframe);
            vi.mocked(parseUrl).mockReturnValue(undefined);
            youtubePlaylist.updateUrl("not-a-valid-url");
            expect(iframe.getAttribute("src")).toBe("https://www.youtube.com/embed/old");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
        it("should handle multiple iframes in element", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe1 = document.createElement("iframe");
            const iframe2 = document.createElement("iframe");
            mockElement.appendChild(iframe1);
            mockElement.appendChild(iframe2);
            vi.mocked(parseUrl).mockReturnValue({
                url: "https://www.youtube.com/playlist?list=PLxyz",
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLxyz",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("https://www.youtube.com/playlist?list=PLxyz");
            expect(iframe1.getAttribute("src")).toBe("https://www.youtube.com/playlist?list=PLxyz");
            expect(iframe2.getAttribute("src")).toBeNull();
        });
        it("should properly replace element content when creating new iframe", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const existingContent = document.createElement("div");
            existingContent.textContent = "Old content";
            mockElement.appendChild(existingContent);
            vi.mocked(parseUrl).mockReturnValue({
                url: "https://www.youtube.com/playlist?list=PLnew",
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLnew",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("https://www.youtube.com/playlist?list=PLnew");
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).not.toBeNull();
            expect(mockElement.textContent).not.toContain("Old content");
        });
        it("should handle YouTube URL with different paths", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            vi.mocked(parseUrl).mockReturnValue({
                url: "https://www.youtube.com/watch?v=abc&list=PLtest",
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/watch",
                search: "?v=abc&list=PLtest",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("https://www.youtube.com/watch?v=abc&list=PLtest");
            expect(iframe.getAttribute("src")).toBe("https://www.youtube.com/watch?v=abc&list=PLtest");
        });
        it("should handle YouTube nocookie embed URL", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            vi.mocked(parseUrl).mockReturnValue({
                url: "https://www.youtube-nocookie.com/embed/xyz",
                protocol: "https:",
                host: "www.youtube-nocookie.com",
                hostname: "www.youtube-nocookie.com",
                port: "",
                path: "/embed/xyz",
                search: "",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("https://www.youtube-nocookie.com/embed/xyz");
            expect(iframe.getAttribute("src")).toBe("https://www.youtube-nocookie.com/embed/xyz");
        });
    });
    describe("Iframe attributes", () => {
        beforeEach(() => {
            vi.mocked(parseUrl).mockImplementation(createMockParseUrl);
        });
        it("should set all required attributes when creating new iframe", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const url = "https://www.youtube.com/playlist?list=PLtest";
            vi.mocked(parseUrl).mockReturnValue({
                url,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLtest",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(url);
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).not.toBeNull();
            expect(iframe.getAttribute("src")).toBe(url);
            expect(iframe.getAttribute("frameborder")).toBe("0");
            expect(iframe.hasAttribute("webkitAllowFullScreen")).toBe(true);
            expect(iframe.hasAttribute("mozallowfullscreen")).toBe(true);
            expect(iframe.hasAttribute("allowfullscreen")).toBe(true);
        });
        it("should only update src attribute on existing iframe", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://www.youtube.com/old");
            iframe.setAttribute("width", "640");
            iframe.setAttribute("height", "480");
            mockElement.appendChild(iframe);
            const newUrl = "https://www.youtube.com/playlist?list=PLnew";
            vi.mocked(parseUrl).mockReturnValue({
                url: newUrl,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLnew",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(newUrl);
            expect(iframe.getAttribute("src")).toBe(newUrl);
            expect(iframe.getAttribute("width")).toBe("640");
            expect(iframe.getAttribute("height")).toBe("480");
        });
    });
    describe("Supported hosts validation", () => {
        beforeEach(() => {
            vi.mocked(parseUrl).mockImplementation(createMockParseUrl);
        });
        it("should support www.youtube.com host", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const url = "https://www.youtube.com/playlist?list=PLtest1";
            vi.mocked(parseUrl).mockReturnValue({
                url,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLtest1",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(url);
            expect(iframe.getAttribute("src")).toBe(url);
            expect(snackbar).not.toHaveBeenCalled();
        });
        it("should support www.youtube-nocookie.com host", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const url = "https://www.youtube-nocookie.com/embed/videoseries?list=PLtest2";
            vi.mocked(parseUrl).mockReturnValue({
                url,
                protocol: "https:",
                host: "www.youtube-nocookie.com",
                hostname: "www.youtube-nocookie.com",
                port: "",
                path: "/embed/videoseries",
                search: "?list=PLtest2",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(url);
            expect(iframe.getAttribute("src")).toBe(url);
            expect(snackbar).not.toHaveBeenCalled();
        });
        it("should reject unsupported hosts", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            vi.mocked(parseUrl).mockReturnValue({
                url: "https://www.vimeo.com/123456",
                protocol: "https:",
                host: "www.vimeo.com",
                hostname: "www.vimeo.com",
                port: "",
                path: "/123456",
                search: "",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("https://www.vimeo.com/123456");
            expect(snackbar).toHaveBeenCalledWith("snackbars.video-service-not-supported", "error");
        });
    });
    describe("Error handling for malformed URLs", () => {
        it("should handle parseUrl returning object with empty host string", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue({
                url: "http://",
                protocol: "http:",
                host: "",
                hostname: "",
                port: "",
                path: "",
                search: "",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl("http://");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
        it("should handle URL with only protocol", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue(undefined);
            youtubePlaylist.updateUrl("https://");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
        it("should not create iframe when URL is invalid", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            vi.mocked(parseUrl).mockReturnValue(undefined);
            youtubePlaylist.updateUrl("invalid-url");
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).toBeNull();
        });
    });
    describe("Complete workflow scenarios", () => {
        beforeEach(() => {
            vi.mocked(parseUrl).mockImplementation(createMockParseUrl);
        });
        it("should handle complete lifecycle: create, activate, click, update, deactivate", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            youtubePlaylist.activate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(false);
            const url = "https://www.youtube.com/playlist?list=PLabc";
            vi.mocked(parseUrl).mockReturnValue({
                url,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLabc",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(url);
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).not.toBeNull();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(true);
            const mockEvent = new MouseEvent("click");
            youtubePlaylist.handleClick(mockEvent);
            expect(mockBlock.manageActions).toHaveBeenCalled();
            youtubePlaylist.desactivate();
            expect(mockElement.contains(youtubePlaylist["mask"])).toBe(false);
        });
        it("should handle updating from one valid URL to another", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const firstUrl = "https://www.youtube.com/playlist?list=PLfirst";
            vi.mocked(parseUrl).mockReturnValue({
                url: firstUrl,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLfirst",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(firstUrl);
            expect(iframe.getAttribute("src")).toBe(firstUrl);
            const secondUrl = "https://www.youtube.com/playlist?list=PLsecond";
            vi.mocked(parseUrl).mockReturnValue({
                url: secondUrl,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLsecond",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(secondUrl);
            expect(iframe.getAttribute("src")).toBe(secondUrl);
        });
        it("should handle switching from valid URL to invalid URL", () => {
            const youtubePlaylist = new YoutubePlaylist(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const validUrl = "https://www.youtube.com/playlist?list=PLvalid";
            vi.mocked(parseUrl).mockReturnValue({
                url: validUrl,
                protocol: "https:",
                host: "www.youtube.com",
                hostname: "www.youtube.com",
                port: "",
                path: "/playlist",
                search: "?list=PLvalid",
                hash: "",
                searchObject: {},
            });
            youtubePlaylist.updateUrl(validUrl);
            expect(iframe.getAttribute("src")).toBe(validUrl);
            vi.mocked(parseUrl).mockReturnValue(undefined);
            youtubePlaylist.updateUrl("invalid-url");
            expect(iframe.getAttribute("src")).toBe(validUrl);
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
        });
    });
});
