import { describe, it, expect, beforeEach, vi, afterEach } from "vitest";
import Video from "../Video";
vi.mock("../../services/url", () => ({
    parseUrl: vi.fn(),
    buildParams: vi.fn(),
}));
vi.mock("../../user-interface/plugin/video-modal", () => ({
    default: vi.fn(),
}));
vi.mock("../../user-interface/common/eden-snackbar", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../../services/html", () => ({
    default: vi.fn((strings, ...values) => {
        return strings.reduce((acc, str, i) => acc + str + (values[i] || ""), "");
    }),
}));
vi.mock("../../i18n/index", () => ({
    default: vi.fn((key) => key),
}));
import { parseUrl, buildParams } from "../../services/url";
import { snackbar } from "../../user-interface/common/eden-snackbar";
import VideoModal from "../../user-interface/plugin/video-modal";
describe("Video Brick", () => {
    let mockBlock;
    let element;
    let parseUrlMock;
    let snackbarMock;
    let buildParamsMock;
    beforeEach(() => {
        element = document.createElement("div");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
        parseUrlMock = vi.mocked(parseUrl);
        snackbarMock = vi.mocked(snackbar);
        buildParamsMock = vi.mocked(buildParams);
        vi.clearAllMocks();
    });
    afterEach(() => {
        element.innerHTML = "";
    });
    describe("constructor", () => {
        it("should initialize Video brick with empty element", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(video.element).toBe(element);
            expect(video.block).toBe(mockBlock);
            expect(video.mask).toBeInstanceOf(HTMLDivElement);
        });
        it("should create mask div with correct styles", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(video.mask.style.position).toBe("absolute");
            expect(video.mask.style.top).toBe("0px");
            expect(video.mask.style.bottom).toBe("0px");
            expect(video.mask.style.left).toBe("0px");
            expect(video.mask.style.right).toBe("0px");
        });
        it("should initialize actions with add-video action", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(video.actions).toHaveLength(1);
            expect(video.actions[0].tooltip).toBe("bricks.video.add-video");
            expect(video.actions[0].icon).toBe("movie");
            expect(typeof video.actions[0].action).toBe("function");
        });
        it("should handle element with existing iframe", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://www.youtube-nocookie.com/embed/test";
            element.appendChild(existingIframe);
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(video.element.querySelector("iframe")).toBe(existingIframe);
        });
        it("should store parent bricks", () => {
            const parentBricks = [
                new Video({ block: mockBlock, element, parentBricks: [] }),
            ];
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks,
            });
            expect(video.parentBricks).toBe(parentBricks);
        });
    });
    describe("actions", () => {
        it("should call VideoModal with empty URL for new video", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = video.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            expect(VideoModal).toHaveBeenCalledWith(video.updateUrl, "");
        });
        it("should call VideoModal with existing video URL", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://www.youtube-nocookie.com/embed/test123";
            element.appendChild(existingIframe);
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = video.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            expect(VideoModal).toHaveBeenCalledWith(video.updateUrl, "https://www.youtube-nocookie.com/embed/test123");
        });
        it("should handle missing iframe gracefully", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => {
                const action = video.actions[0].action;
                if (typeof action === "function") {
                    action({});
                }
            }).not.toThrow();
        });
    });
    describe("updateUrl", () => {
        describe("YouTube URLs", () => {
            it("should convert standard YouTube URL to nocookie embed", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://www.youtube.com/watch?v=test123",
                    host: "www.youtube.com",
                    protocol: "https:",
                    hostname: "www.youtube.com",
                    port: "",
                    path: "/watch",
                    search: "?v=test123",
                    hash: "",
                    searchObject: { v: "test123" },
                });
                buildParamsMock.mockReturnValue("?rel=0");
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://www.youtube.com/watch?v=test123");
                const iframe = element.querySelector("iframe");
                expect(iframe).toBeTruthy();
                expect(iframe?.getAttribute("src")).toBe("https://www.youtube-nocookie.com/embed/test123?rel=0");
            });
            it("should handle YouTube embed URL", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://www.youtube.com/embed/test456",
                    host: "www.youtube.com",
                    protocol: "https:",
                    hostname: "www.youtube.com",
                    port: "",
                    path: "/embed/test456",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                buildParamsMock.mockReturnValue("?rel=0");
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://www.youtube.com/embed/test456");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("src")).toBe("https://www.youtube-nocookie.com/embed/test456?rel=0");
            });
            it("should handle youtube-nocookie.com URLs", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://www.youtube-nocookie.com/embed/test789",
                    host: "www.youtube-nocookie.com",
                    protocol: "https:",
                    hostname: "www.youtube-nocookie.com",
                    port: "",
                    path: "/embed/test789",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                buildParamsMock.mockReturnValue("?rel=0");
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://www.youtube-nocookie.com/embed/test789");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("src")).toBe("https://www.youtube-nocookie.com/embed/test789?rel=0");
            });
            it("should handle youtu.be short URLs", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://youtu.be/short123",
                    host: "youtu.be",
                    protocol: "https:",
                    hostname: "youtu.be",
                    port: "",
                    path: "/short123",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://youtu.be/short123");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("src")).toBe("https://www.youtube-nocookie.com/embed/short123");
            });
            it("should preserve additional YouTube parameters", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://www.youtube.com/watch?v=test123&t=30s",
                    host: "www.youtube.com",
                    protocol: "https:",
                    hostname: "www.youtube.com",
                    port: "",
                    path: "/watch",
                    search: "?v=test123&t=30s",
                    hash: "",
                    searchObject: { v: "test123", t: "30s" },
                });
                buildParamsMock.mockReturnValue("?rel=0&t=30s");
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://www.youtube.com/watch?v=test123&t=30s");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("src")).toBe("https://www.youtube-nocookie.com/embed/test123?rel=0&t=30s");
            });
            it("should show error for invalid YouTube URL format", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://www.youtube.com/invalid",
                    host: "www.youtube.com",
                    protocol: "https:",
                    hostname: "www.youtube.com",
                    port: "",
                    path: "/invalid",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://www.youtube.com/invalid");
                expect(snackbarMock).toHaveBeenCalledWith("snackbars.wrong-url-format", "error");
                expect(element.querySelector("iframe")).toBeNull();
            });
        });
        describe("Vimeo URLs", () => {
            it("should convert Vimeo URL to embed", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://vimeo.com/123456789",
                    host: "vimeo.com",
                    protocol: "https:",
                    hostname: "vimeo.com",
                    port: "",
                    path: "/123456789",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://vimeo.com/123456789");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("src")).toBe("https://player.vimeo.com/video/123456789");
            });
            it("should handle Vimeo player embed URL", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://player.vimeo.com/video/987654321",
                    host: "player.vimeo.com",
                    protocol: "https:",
                    hostname: "player.vimeo.com",
                    port: "",
                    path: "/video/987654321",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://player.vimeo.com/video/987654321");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("src")).toBe("https://player.vimeo.com/video/987654321");
            });
        });
        describe("Dailymotion URLs", () => {
            it("should convert Dailymotion URL to embed", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://www.dailymotion.com/video/x8abcd",
                    host: "www.dailymotion.com",
                    protocol: "https:",
                    hostname: "www.dailymotion.com",
                    port: "",
                    path: "/video/x8abcd",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://www.dailymotion.com/video/x8abcd");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("src")).toBe("https://www.dailymotion.com/embed/video/x8abcd");
            });
        });
        describe("iframe creation", () => {
            it("should create iframe with correct attributes", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://vimeo.com/123456789",
                    host: "vimeo.com",
                    protocol: "https:",
                    hostname: "vimeo.com",
                    port: "",
                    path: "/123456789",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://vimeo.com/123456789");
                const iframe = element.querySelector("iframe");
                expect(iframe?.getAttribute("frameborder")).toBe("0");
                expect(iframe?.hasAttribute("webkitAllowFullScreen")).toBe(true);
                expect(iframe?.hasAttribute("mozallowfullscreen")).toBe(true);
                expect(iframe?.hasAttribute("allowfullscreen")).toBe(true);
            });
            it("should append mask after creating iframe", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://vimeo.com/123456789",
                    host: "vimeo.com",
                    protocol: "https:",
                    hostname: "vimeo.com",
                    port: "",
                    path: "/123456789",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://vimeo.com/123456789");
                expect(element.contains(video.mask)).toBe(true);
            });
            it("should update existing iframe src", () => {
                const existingIframe = document.createElement("iframe");
                existingIframe.src = "https://www.youtube-nocookie.com/embed/old";
                element.appendChild(existingIframe);
                parseUrlMock.mockReturnValue({
                    url: "https://vimeo.com/123456789",
                    host: "vimeo.com",
                    protocol: "https:",
                    hostname: "vimeo.com",
                    port: "",
                    path: "/123456789",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://vimeo.com/123456789");
                expect(existingIframe.src).toBe("https://player.vimeo.com/video/123456789");
                expect(element.querySelectorAll("iframe")).toHaveLength(1);
            });
        });
        describe("error handling", () => {
            it("should throw error for unsupported video service (bug)", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://unsupported.com/video/123",
                    host: "unsupported.com",
                    protocol: "https:",
                    hostname: "unsupported.com",
                    port: "",
                    path: "/video/123",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                expect(() => {
                    video.updateUrl("https://unsupported.com/video/123");
                }).toThrow(TypeError);
                expect(element.querySelector("iframe")).toBeNull();
            });
            it("should show error for invalid URL", () => {
                parseUrlMock.mockReturnValue(undefined);
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("invalid-url");
                expect(snackbarMock).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
                expect(element.querySelector("iframe")).toBeNull();
            });
            it("should show error for URL without host", () => {
                parseUrlMock.mockReturnValue({
                    url: "https://example.com",
                    host: "",
                    protocol: "https:",
                    hostname: "",
                    port: "",
                    path: "",
                    search: "",
                    hash: "",
                    searchObject: {},
                });
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("https://example.com");
                expect(snackbarMock).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
            });
            it("should show error for null parseUrl result", () => {
                parseUrlMock.mockReturnValue(null);
                const video = new Video({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
                video.updateUrl("bad-url");
                expect(snackbarMock).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
            });
        });
    });
    describe("activate", () => {
        it("should add click event listener", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const addEventListenerSpy = vi.spyOn(element, "addEventListener");
            video.activate();
            expect(addEventListenerSpy).toHaveBeenCalledWith("click", video.handleClick);
        });
        it("should append mask when iframe exists", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://www.youtube-nocookie.com/embed/test";
            element.appendChild(existingIframe);
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.activate();
            expect(element.contains(video.mask)).toBe(true);
        });
        it("should not append mask when no iframe exists", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.activate();
            expect(element.contains(video.mask)).toBe(false);
        });
        it("should handle null element gracefully", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.element = null;
            expect(() => video.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const removeEventListenerSpy = vi.spyOn(element, "removeEventListener");
            video.activate();
            video.desactivate();
            expect(removeEventListenerSpy).toHaveBeenCalledWith("click", video.handleClick);
        });
        it("should remove mask from element", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://www.youtube-nocookie.com/embed/test";
            element.appendChild(existingIframe);
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.activate();
            expect(element.contains(video.mask)).toBe(true);
            video.desactivate();
            expect(element.contains(video.mask)).toBe(false);
        });
        it("should handle null element gracefully", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.element = null;
            expect(() => video.desactivate()).not.toThrow();
        });
        it("should handle mask removal when not in DOM", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => video.desactivate()).not.toThrow();
        });
    });
    describe("lifecycle", () => {
        it("should handle complete activate/desactivate cycle", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://www.youtube-nocookie.com/embed/test";
            element.appendChild(existingIframe);
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.activate();
            expect(element.contains(video.mask)).toBe(true);
            video.desactivate();
            expect(element.contains(video.mask)).toBe(false);
            video.activate();
            expect(element.contains(video.mask)).toBe(true);
        });
        it("should handle multiple activate calls", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://www.youtube-nocookie.com/embed/test";
            element.appendChild(existingIframe);
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.activate();
            video.activate();
            const masks = Array.from(element.children).filter((child) => child === video.mask);
            expect(masks).toHaveLength(1);
        });
    });
    describe("handleClick", () => {
        it("should call block.manageActions with correct bricks", () => {
            const parentBricks = [
                new Video({ block: mockBlock, element, parentBricks: [] }),
            ];
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks,
            });
            video.handleClick();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([
                ...parentBricks,
                video,
            ]);
        });
        it("should prevent default and stop propagation on click event", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const mockEvent = {
                preventDefault: vi.fn(),
                stopPropagation: vi.fn(),
            };
            video.handleClick(mockEvent);
            expect(mockEvent.preventDefault).toHaveBeenCalled();
            expect(mockEvent.stopPropagation).toHaveBeenCalled();
        });
        it("should work without parentBricks", () => {
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.handleClick();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([video]);
        });
    });
    describe("edge cases", () => {
        it("should handle YouTube URL with empty video ID", () => {
            parseUrlMock.mockReturnValue({
                url: "https://www.youtube.com/watch?v=",
                host: "www.youtube.com",
                protocol: "https:",
                hostname: "www.youtube.com",
                port: "",
                path: "/watch",
                search: "?v=",
                hash: "",
                searchObject: { v: "" },
            });
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.updateUrl("https://www.youtube.com/watch?v=");
            expect(snackbarMock).toHaveBeenCalledWith("snackbars.wrong-url-format", "error");
        });
        it("should handle YouTube URL without video parameter", () => {
            parseUrlMock.mockReturnValue({
                url: "https://www.youtube.com/watch",
                host: "www.youtube.com",
                protocol: "https:",
                hostname: "www.youtube.com",
                port: "",
                path: "/watch",
                search: "",
                hash: "",
                searchObject: {},
            });
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.updateUrl("https://www.youtube.com/watch");
            expect(snackbarMock).toHaveBeenCalledWith("snackbars.wrong-url-format", "error");
        });
        it("should handle YouTube URL with short video ID (single char)", () => {
            parseUrlMock.mockReturnValue({
                url: "https://www.youtube.com/watch?v=a",
                host: "www.youtube.com",
                protocol: "https:",
                hostname: "www.youtube.com",
                port: "",
                path: "/watch",
                search: "?v=a",
                hash: "",
                searchObject: { v: "a" },
            });
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.updateUrl("https://www.youtube.com/watch?v=a");
            expect(snackbarMock).toHaveBeenCalledWith("snackbars.wrong-url-format", "error");
        });
        it("should handle concurrent updateUrl calls", () => {
            parseUrlMock.mockReturnValue({
                url: "https://vimeo.com/123",
                host: "vimeo.com",
                protocol: "https:",
                hostname: "vimeo.com",
                port: "",
                path: "/123",
                search: "",
                hash: "",
                searchObject: {},
            });
            const video = new Video({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            video.updateUrl("https://vimeo.com/123");
            parseUrlMock.mockReturnValue({
                url: "https://vimeo.com/456",
                host: "vimeo.com",
                protocol: "https:",
                hostname: "vimeo.com",
                port: "",
                path: "/456",
                search: "",
                hash: "",
                searchObject: {},
            });
            video.updateUrl("https://vimeo.com/456");
            const iframes = element.querySelectorAll("iframe");
            expect(iframes).toHaveLength(1);
            expect(iframes[0].src).toBe("https://player.vimeo.com/video/456");
        });
    });
});
