import { describe, it, expect, beforeEach, vi } from "vitest";
import Slideshow from "../Slideshow";
vi.mock("../../user-interface/plugin/slideshow-modal", () => ({
    default: class MockSlideshowModal {
        constructor(callback, images, width, height, credits) {
            this.callback = callback;
            this.images = images;
            this.width = width;
            this.height = height;
            this.credits = credits;
        }
    },
}));
describe("Slideshow Brick", () => {
    let mockBlock;
    let element;
    beforeEach(() => {
        element = document.createElement("div");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
    });
    describe("constructor", () => {
        it("should initialize Slideshow brick", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(slideshow.element).toBe(element);
            expect(slideshow.actions.length).toBe(1);
        });
        it("should store markup when provided", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            expect(slideshow.markup).toBe(markup);
        });
        it("should have undefined markup when not provided", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(slideshow.markup).toBeUndefined();
        });
        it("should create action with correct properties", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(slideshow.actions[0]).toMatchObject({
                icon: "slideshow",
            });
            expect(slideshow.actions[0].tooltip).toBeDefined();
            expect(typeof slideshow.actions[0].action).toBe("function");
        });
    });
    describe("action callback", () => {
        it("should extract images data from element", () => {
            const img1 = document.createElement("img");
            img1.src = "image1.jpg";
            img1.alt = "Image 1";
            img1.setAttribute("data-metadata", "metadata1");
            const img2 = document.createElement("img");
            img2.src = "image2.jpg";
            img2.alt = "Image 2";
            element.appendChild(img1);
            element.appendChild(img2);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const SlideshowModalMock = vi.fn();
            vi.doMock("../../user-interface/plugin/slideshow-modal", () => ({
                default: SlideshowModalMock,
            }));
            slideshow.actions[0].action();
            const imgs = Array.from(element.querySelectorAll("img")).map((img) => ({
                src: img.getAttribute("src") || "",
                alt: img.getAttribute("alt") || "",
                metadata: img.getAttribute("data-metadata"),
            }));
            expect(imgs).toEqual([
                { src: "image1.jpg", alt: "Image 1", metadata: "metadata1" },
                { src: "image2.jpg", alt: "Image 2", metadata: null },
            ]);
        });
        it("should handle images without metadata attribute", () => {
            const img = document.createElement("img");
            img.src = "image.jpg";
            img.alt = "Test";
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            const imgs = Array.from(element.querySelectorAll("img")).map((img) => ({
                src: img.getAttribute("src") || "",
                alt: img.getAttribute("alt") || "",
                metadata: img.getAttribute("data-metadata"),
            }));
            expect(imgs[0].metadata).toBeNull();
        });
        it("should extract credits from element", () => {
            const img = document.createElement("img");
            img.src = "image.jpg";
            element.appendChild(img);
            element.setAttribute("data-credits", "Photo by John Doe");
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            expect(element.getAttribute("data-credits")).toBe("Photo by John Doe");
        });
        it("should handle missing credits attribute", () => {
            const img = document.createElement("img");
            img.src = "image.jpg";
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            expect(element.getAttribute("data-credits")).toBeNull();
        });
        it("should extract width and height from img element", () => {
            const img = document.createElement("img");
            img.src = "image.jpg";
            img.setAttribute("width", "800");
            img.setAttribute("height", "600");
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            expect(img.getAttribute("width")).toBe("800");
            expect(img.getAttribute("height")).toBe("600");
        });
        it("should handle missing width and height attributes", () => {
            const img = document.createElement("img");
            img.src = "image.jpg";
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            expect(img.hasAttribute("width")).toBe(false);
            expect(img.hasAttribute("height")).toBe(false);
        });
        it("should return early if no img element exists", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => slideshow.actions[0].action()).not.toThrow();
        });
        it("should handle empty src and alt attributes", () => {
            const img = document.createElement("img");
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            const imgs = Array.from(element.querySelectorAll("img")).map((img) => ({
                src: img.getAttribute("src") || "",
                alt: img.getAttribute("alt") || "",
                metadata: img.getAttribute("data-metadata"),
            }));
            expect(imgs[0]).toEqual({ src: "", alt: "", metadata: null });
        });
    });
    describe("createSlideShow", () => {
        it("should clear element innerHTML", () => {
            element.innerHTML = "<p>Old content</p>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup: "<img src='' alt=''>",
                parentBricks: [],
            });
            const images = [{ src: "new.jpg", alt: "New", metadata: null }];
            slideshow.createSlideShow(images, "");
            expect(element.querySelector("p")).toBeNull();
        });
        it("should create img elements from images array", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [
                { src: "image1.jpg", alt: "Image 1", metadata: null },
                { src: "image2.jpg", alt: "Image 2", metadata: null },
            ];
            slideshow.createSlideShow(images, "");
            const imgs = element.querySelectorAll("img");
            expect(imgs.length).toBe(2);
            expect(imgs[0].src).toContain("image1.jpg");
            expect(imgs[1].src).toContain("image2.jpg");
        });
        it("should set src attribute on images", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            const img = element.querySelector("img");
            expect(img?.getAttribute("src")).toBe("test.jpg");
        });
        it("should set alt attribute on images", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [{ src: "test.jpg", alt: "Alt text", metadata: null }];
            slideshow.createSlideShow(images, "");
            const img = element.querySelector("img");
            expect(img?.getAttribute("alt")).toBe("Alt text");
        });
        it("should set metadata attribute when provided", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [
                { src: "test.jpg", alt: "Test", metadata: "metadata-value" },
            ];
            slideshow.createSlideShow(images, "");
            const img = element.querySelector("img");
            expect(img?.getAttribute("data-metadata")).toBe("metadata-value");
        });
        it("should not set metadata attribute when null", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            const img = element.querySelector("img");
            expect(img?.hasAttribute("data-metadata")).toBe(false);
        });
        it("should handle markup with wrapper elements", () => {
            const markup = "<div><img src='' alt=''></div>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            const img = element.querySelector("img");
            expect(img).toBeTruthy();
            expect(img?.getAttribute("src")).toBe("test.jpg");
        });
        it("should return early if markup is undefined", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            expect(element.querySelectorAll("img").length).toBe(0);
        });
        it("should return early if no img node found in markup", () => {
            const markup = "<div></div>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            const divs = element.querySelectorAll("div");
            expect(divs.length).toBe(1);
            expect(element.querySelector("img")).toBeNull();
        });
        it("should call updateCredits with provided credits", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const updateCreditsSpy = vi.spyOn(slideshow, "updateCredits");
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "Photo by Jane");
            expect(updateCreditsSpy).toHaveBeenCalledWith("Photo by Jane");
        });
        it("should call desactivate before reactivating", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const desactivateSpy = vi.spyOn(slideshow, "desactivate");
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            expect(desactivateSpy).toHaveBeenCalled();
        });
        it("should call activate after creating slideshow", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const activateSpy = vi.spyOn(slideshow, "activate");
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            expect(activateSpy).toHaveBeenCalled();
        });
        it("should handle multiple images correctly", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [
                { src: "img1.jpg", alt: "Alt 1", metadata: "meta1" },
                { src: "img2.jpg", alt: "Alt 2", metadata: null },
                { src: "img3.jpg", alt: "Alt 3", metadata: "meta3" },
            ];
            slideshow.createSlideShow(images, "Credits");
            const imgs = element.querySelectorAll("img");
            expect(imgs.length).toBe(3);
            expect(imgs[0].getAttribute("src")).toBe("img1.jpg");
            expect(imgs[0].getAttribute("alt")).toBe("Alt 1");
            expect(imgs[0].getAttribute("data-metadata")).toBe("meta1");
            expect(imgs[1].getAttribute("src")).toBe("img2.jpg");
            expect(imgs[2].getAttribute("src")).toBe("img3.jpg");
            expect(imgs[2].getAttribute("data-metadata")).toBe("meta3");
        });
    });
    describe("updateCredits", () => {
        it("should set data-credits attribute when text is provided", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.updateCredits("Photo by John");
            expect(element.getAttribute("data-credits")).toBe("Photo by John");
        });
        it("should remove data-credits attribute when text is empty", () => {
            element.setAttribute("data-credits", "Old credits");
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.updateCredits("");
            expect(element.hasAttribute("data-credits")).toBe(false);
        });
        it("should update existing credits", () => {
            element.setAttribute("data-credits", "Old credits");
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.updateCredits("New credits");
            expect(element.getAttribute("data-credits")).toBe("New credits");
        });
        it("should handle whitespace-only text as truthy", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.updateCredits("   ");
            expect(element.getAttribute("data-credits")).toBe("   ");
        });
    });
    describe("activate", () => {
        it("should add click event listener to element", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const addEventListenerSpy = vi.spyOn(element, "addEventListener");
            slideshow.activate();
            expect(addEventListenerSpy).toHaveBeenCalledWith("click", slideshow.handleClick);
        });
        it("should handle activation when element exists", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => slideshow.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener from element", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const removeEventListenerSpy = vi.spyOn(element, "removeEventListener");
            slideshow.desactivate();
            expect(removeEventListenerSpy).toHaveBeenCalledWith("click", slideshow.handleClick);
        });
        it("should handle deactivation when element exists", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => slideshow.desactivate()).not.toThrow();
        });
        it("should allow multiple desactivate calls", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.activate();
            slideshow.desactivate();
            expect(() => slideshow.desactivate()).not.toThrow();
        });
    });
    describe("event handling flow", () => {
        it("should add and remove event listeners correctly", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const addSpy = vi.spyOn(element, "addEventListener");
            const removeSpy = vi.spyOn(element, "removeEventListener");
            slideshow.activate();
            expect(addSpy).toHaveBeenCalledWith("click", slideshow.handleClick);
            slideshow.desactivate();
            expect(removeSpy).toHaveBeenCalledWith("click", slideshow.handleClick);
        });
        it("should maintain same handleClick reference for activate/desactivate", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const handler1 = slideshow.handleClick;
            slideshow.activate();
            slideshow.desactivate();
            const handler2 = slideshow.handleClick;
            expect(handler1).toBe(handler2);
        });
    });
    describe("integration scenarios", () => {
        it("should handle complete workflow: create, update credits, activate", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [
                { src: "photo1.jpg", alt: "Photo 1", metadata: "data1" },
                { src: "photo2.jpg", alt: "Photo 2", metadata: null },
            ];
            slideshow.createSlideShow(images, "Photographer: Alice");
            const imgs = element.querySelectorAll("img");
            expect(imgs.length).toBe(2);
            expect(element.getAttribute("data-credits")).toBe("Photographer: Alice");
        });
        it("should replace existing slideshow with new images", () => {
            const markup = "<img src='' alt=''>";
            const img = document.createElement("img");
            img.src = "old.jpg";
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const newImages = [
                { src: "new1.jpg", alt: "New 1", metadata: null },
                { src: "new2.jpg", alt: "New 2", metadata: null },
            ];
            slideshow.createSlideShow(newImages, "New credits");
            const imgs = element.querySelectorAll("img");
            expect(imgs.length).toBe(2);
            expect(imgs[0].src).toContain("new1.jpg");
            expect(imgs[1].src).toContain("new2.jpg");
            expect(element.querySelector("img[src*='old.jpg']")).toBeNull();
        });
        it("should handle empty images array", () => {
            const markup = "<img src='' alt=''>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            slideshow.createSlideShow([], "Credits");
            expect(element.querySelectorAll("img").length).toBe(0);
            expect(element.getAttribute("data-credits")).toBe("Credits");
        });
        it("should handle action with nested img in wrapper", () => {
            const wrapper = document.createElement("figure");
            const img = document.createElement("img");
            img.src = "nested.jpg";
            img.alt = "Nested";
            img.setAttribute("width", "500");
            img.setAttribute("height", "400");
            wrapper.appendChild(img);
            element.appendChild(wrapper);
            element.setAttribute("data-credits", "Test Credits");
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            expect(element.querySelector("img")).toBe(img);
        });
    });
    describe("edge cases", () => {
        it("should handle image without src attribute", () => {
            const img = document.createElement("img");
            img.alt = "No src";
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.actions[0].action();
            const imgs = Array.from(element.querySelectorAll("img")).map((img) => ({
                src: img.getAttribute("src") || "",
                alt: img.getAttribute("alt") || "",
                metadata: img.getAttribute("data-metadata"),
            }));
            expect(imgs[0].src).toBe("");
        });
        it("should handle invalid width and height values", () => {
            const img = document.createElement("img");
            img.src = "test.jpg";
            img.setAttribute("width", "invalid");
            img.setAttribute("height", "also-invalid");
            element.appendChild(img);
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => slideshow.actions[0].action()).not.toThrow();
        });
        it("should handle getAttribute returning null for width and height", () => {
            const img = document.createElement("img");
            img.src = "test.jpg";
            element.appendChild(img);
            const originalGetAttribute = img.getAttribute.bind(img);
            img.getAttribute = vi.fn((attr) => {
                if (attr === "width" || attr === "height") {
                    return null;
                }
                return originalGetAttribute(attr);
            });
            img.setAttribute("width", "800");
            img.setAttribute("height", "600");
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => slideshow.actions[0].action()).not.toThrow();
        });
        it("should handle markup that is just an img tag", () => {
            const markup = "<img src='template.jpg' alt='Template' width='100'>";
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            const images = [{ src: "test.jpg", alt: "Test", metadata: null }];
            slideshow.createSlideShow(images, "");
            const img = element.querySelector("img");
            expect(img?.nodeName).toBe("IMG");
            expect(img?.getAttribute("src")).toBe("test.jpg");
        });
        it("should preserve other element attributes after updateCredits", () => {
            element.setAttribute("data-other", "value");
            element.setAttribute("class", "slideshow");
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            slideshow.updateCredits("Credits");
            expect(element.getAttribute("data-other")).toBe("value");
            expect(element.getAttribute("class")).toBe("slideshow");
        });
        it("should handle rapid activate/deactivate cycles", () => {
            const slideshow = new Slideshow({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => {
                slideshow.activate();
                slideshow.desactivate();
                slideshow.activate();
                slideshow.desactivate();
                slideshow.activate();
            }).not.toThrow();
        });
    });
});
