import { describe, it, expect, beforeEach, vi } from "vitest";
import Icon from "../Icon";
vi.mock("../../user-interface/common/eden-modal", () => ({
    default: class MockEdenModal {
    },
}));
vi.mock("../../user-interface/common/eden-snackbar", () => ({
    snackbar: vi.fn(),
}));
describe("Icon Brick", () => {
    let mockBlock;
    let element;
    beforeEach(() => {
        element = document.createElement("div");
        const svg = document.createElementNS("http://www.w3.org/2000/svg", "svg");
        const use = document.createElementNS("http://www.w3.org/2000/svg", "use");
        use.setAttribute("href", "#icon-test");
        svg.appendChild(use);
        element.appendChild(svg);
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
        window.EdenConfig = {
            iconsSprite: "/sprites.svg",
        };
        global.fetch = vi.fn().mockResolvedValue({
            text: () => Promise.resolve('<svg><symbol id="icon-test"></symbol></svg>'),
            json: () => Promise.resolve([]),
        });
    });
    describe("constructor", () => {
        it("should initialize Icon brick", () => {
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(icon.element).toBe(element);
            expect(icon.actions.length).toBeGreaterThan(0);
        });
        it("should determine mode from sprite URL", () => {
            window.EdenConfig.iconsSprite = "/sprites.svg";
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(icon.mode).toBe("sprite");
        });
        it("should detect index mode for JSON sprites", () => {
            window.EdenConfig.iconsSprite = "/icons/index.json";
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(icon.mode).toBe("index");
        });
        it("should parse base path from sprite URL", () => {
            window.EdenConfig.iconsSprite = "/assets/icons/sprites.svg";
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(icon.basePath).toBe("/assets/icons");
        });
        it("should create edit action", () => {
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const editAction = icon.actions.find((action) => action.icon === "pen");
            expect(editAction).toBeDefined();
        });
    });
    describe("initElement", () => {
        it("should fetch sprite for sprite mode", async () => {
            const mockSvg = '<svg><symbol id="icon-test"></symbol></svg>';
            global.fetch.mockResolvedValue({
                text: () => Promise.resolve(mockSvg),
            });
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            await icon.initElement();
            expect(global.fetch).toHaveBeenCalledWith("/sprites.svg");
        });
        it("should fetch JSON for index mode", async () => {
            window.EdenConfig.iconsSprite = "/icons/index.json";
            const mockIndex = ["icon1.svg", "icon2.svg", "icon3.svg"];
            global.fetch.mockResolvedValue({
                json: () => Promise.resolve(mockIndex),
            });
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            await icon.initElement();
            expect(global.fetch).toHaveBeenCalledWith("/icons/index.json");
        });
        it("should handle fetch errors gracefully", async () => {
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            global.fetch.mockRejectedValueOnce(new Error("Network error"));
            await expect(icon.initElement()).rejects.toThrow("Network error");
        });
    });
    describe("updateIcon", () => {
        it("should update use element href", () => {
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const use = element.querySelector("use");
            const initialHref = use?.getAttribute("href");
            icon.updateIcon("icon-new");
            const newHref = use?.getAttribute("href");
            expect(newHref).toBe("/sprites.svg#icon-new");
            expect(newHref).not.toBe(initialHref);
        });
        it("should handle missing use element", () => {
            const emptyElement = document.createElement("div");
            const icon = new Icon({
                block: mockBlock,
                element: emptyElement,
                parentBricks: [],
            });
            expect(() => icon.updateIcon("icon-test")).not.toThrow();
        });
        it("should handle undefined id", () => {
            const icon = new Icon({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const use = element.querySelector("use");
            const originalHref = use?.getAttribute("href");
            icon.updateIcon(undefined);
            expect(use?.getAttribute("href")).toBe(originalHref);
        });
    });
});
