import { describe, it, expect, beforeEach, vi } from "vitest";
import GoogleMaps from "../GoogleMaps";
vi.mock("../../user-interface/plugin/google-maps-modal", () => ({
    default: vi.fn(),
}));
import GoogleMapsModal from "../../user-interface/plugin/google-maps-modal";
describe("GoogleMaps Brick", () => {
    let mockBlock;
    let element;
    beforeEach(() => {
        element = document.createElement("div");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
        vi.clearAllMocks();
    });
    describe("constructor", () => {
        it("should initialize GoogleMaps brick", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(googleMaps.element).toBe(element);
            expect(googleMaps.block).toBe(mockBlock);
        });
        it("should create actions with map icon", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(googleMaps.actions).toHaveLength(1);
            expect(googleMaps.actions[0].icon).toBe("map");
            expect(googleMaps.actions[0].tooltip).toBeDefined();
        });
        it("should create mask element", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.activate();
            const mask = element.querySelector("div");
            expect(mask).toBeDefined();
            googleMaps.desactivate();
        });
    });
    describe("openModal", () => {
        it("should open modal with iframe attributes", () => {
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://maps.google.com/test");
            iframe.setAttribute("width", "800");
            iframe.setAttribute("height", "600");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            expect(GoogleMapsModal).toHaveBeenCalledWith(expect.any(Function), "https://maps.google.com/test", "800", "600");
        });
        it("should use default values when iframe has no attributes", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            expect(GoogleMapsModal).toHaveBeenCalledWith(expect.any(Function), "", "1000", "450");
        });
        it("should handle missing iframe", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            expect(GoogleMapsModal).toHaveBeenCalledWith(expect.any(Function), "", "1000", "450");
        });
    });
    describe("updateUrl", () => {
        it("should update existing iframe attributes", () => {
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://old-url.com");
            iframe.setAttribute("width", "500");
            iframe.setAttribute("height", "300");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            const callback = GoogleMapsModal.mock.calls[0][0];
            callback("https://new-url.com", "900", "700");
            expect(iframe.getAttribute("src")).toBe("https://new-url.com");
            expect(iframe.getAttribute("width")).toBe("900");
            expect(iframe.getAttribute("height")).toBe("700");
        });
        it("should create new iframe if none exists", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            const callback = GoogleMapsModal.mock.calls[0][0];
            callback("https://new-url.com", "800", "600");
            const iframe = element.querySelector("iframe");
            expect(iframe).toBeDefined();
            expect(iframe?.getAttribute("src")).toBe("https://new-url.com");
            expect(iframe?.getAttribute("width")).toBe("800");
            expect(iframe?.getAttribute("height")).toBe("600");
        });
    });
    describe("createAndAppendIframe", () => {
        it("should create iframe with correct attributes", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            const callback = GoogleMapsModal.mock.calls[0][0];
            callback("https://maps.test.com", "1000", "500");
            const iframe = element.querySelector("iframe");
            expect(iframe?.src).toContain("maps.test.com");
            expect(iframe?.getAttribute("width")).toBe("1000");
            expect(iframe?.getAttribute("height")).toBe("500");
            expect(iframe?.getAttribute("frameborder")).toBe("0");
            expect(iframe?.getAttribute("allowtransparency")).toBe("true");
            expect(iframe?.getAttribute("allow")).toBe("encrypted-media");
            expect(iframe?.style.maxWidth).toBe("100%");
        });
        it("should append mask after creating iframe", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            const callback = GoogleMapsModal.mock.calls[0][0];
            callback("https://maps.test.com", "800", "600");
            const children = Array.from(element.children);
            const divs = children.filter((child) => child.tagName === "DIV");
            expect(divs.length).toBeGreaterThan(0);
        });
    });
    describe("activate", () => {
        it("should add click event listener", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const spy = vi.spyOn(element, "addEventListener");
            googleMaps.activate();
            expect(spy).toHaveBeenCalledWith("click", googleMaps.handleClick);
        });
        it("should set element position to relative when iframe exists", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.activate();
            expect(element.style.position).toBe("relative");
        });
        it("should append mask when iframe exists", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.activate();
            const mask = element.querySelector("div");
            expect(mask).toBeDefined();
            expect(mask?.style.position).toBe("absolute");
            expect(mask?.style.top).toBe("0px");
            expect(mask?.style.left).toBe("0px");
            expect(mask?.style.width).toBe("100%");
            expect(mask?.style.height).toBe("100%");
        });
        it("should not throw if element is null", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.element = null;
            expect(() => googleMaps.activate()).not.toThrow();
        });
        it("should handle missing iframe gracefully", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => googleMaps.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.activate();
            const spy = vi.spyOn(element, "removeEventListener");
            googleMaps.desactivate();
            expect(spy).toHaveBeenCalledWith("click", googleMaps.handleClick);
        });
        it("should remove mask", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.activate();
            const maskBefore = element.querySelector("div");
            expect(maskBefore).toBeDefined();
            googleMaps.desactivate();
            const maskAfter = element.querySelector("div");
            expect(maskAfter).toBeNull();
        });
        it("should remove style attribute", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.activate();
            expect(element.hasAttribute("style")).toBe(true);
            googleMaps.desactivate();
            expect(element.hasAttribute("style")).toBe(false);
        });
        it("should not throw if element is null", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.element = null;
            expect(() => googleMaps.desactivate()).not.toThrow();
        });
    });
    describe("integration", () => {
        it("should handle full workflow from opening modal to updating iframe", () => {
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const action = googleMaps.actions[0].action;
            if (typeof action === "function") {
                action({});
            }
            const callback = GoogleMapsModal.mock.calls[0][0];
            callback("https://maps.google.com/embed/v1", "1200", "800");
            const iframe = element.querySelector("iframe");
            expect(iframe).toBeDefined();
            expect(iframe?.getAttribute("src")).toBe("https://maps.google.com/embed/v1");
            googleMaps.activate();
            const mask = element.querySelector("div");
            expect(mask).toBeDefined();
            googleMaps.desactivate();
            const maskAfter = element.querySelector("div");
            expect(maskAfter).toBeNull();
        });
        it("should handle multiple activate/desactivate cycles", () => {
            const iframe = document.createElement("iframe");
            element.appendChild(iframe);
            const googleMaps = new GoogleMaps({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            googleMaps.activate();
            expect(element.querySelector("div")).toBeDefined();
            googleMaps.desactivate();
            expect(element.querySelector("div")).toBeNull();
            googleMaps.activate();
            expect(element.querySelector("div")).toBeDefined();
            googleMaps.desactivate();
            expect(element.querySelector("div")).toBeNull();
        });
    });
});
