import Brick from "../core/brick.js";
import { buildParams, parseUrl } from "../services/url.js";
import VideoModal from "../user-interface/plugin/video-modal.js";
import { snackbar } from "../user-interface/common/eden-snackbar.js";
import html from "../services/html.js";
import t from "../i18n/index.js";
export default class Video extends Brick {
    constructor({ block, element, parentBricks }) {
        super({ block, element, parentBricks });
        this.updateUrl = (url) => {
            const parsedUrl = parseUrl(url);
            if (!parsedUrl?.host) {
                snackbar(t("snackbars.fill-valid-url"), "error");
                return;
            }
            const embedUrl = this.getEmbedUrl(parsedUrl);
            if (embedUrl === undefined) {
                snackbar(t("snackbars.video-service-not-supported"), "error");
                return;
            }
            this.setIframeUrl(embedUrl);
        };
        this.activate = () => {
            if (!this.element)
                return;
            this.element.addEventListener("click", this.handleClick);
            const iframe = this.element.querySelector("iframe");
            if (iframe) {
                this.element.appendChild(this.mask);
            }
        };
        this.desactivate = () => {
            if (!this.element)
                return;
            this.element.removeEventListener("click", this.handleClick);
            this.mask.remove();
        };
        this.mask = this.createMask();
        this.actions = [
            {
                tooltip: t("bricks.video.add-video"),
                icon: "movie",
                action: this.handleAddVideoAction.bind(this),
            },
        ];
    }
    createMask() {
        return Object.assign(document.createElement("div"), {
            style: `
        position: absolute;
        top: 0;
        bottom: 0;
        left: 0;
        right: 0;
      `,
        });
    }
    handleAddVideoAction() {
        const iframe = this.element?.querySelector("iframe");
        const videoUrl = iframe?.getAttribute("src") ?? "";
        new VideoModal(this.updateUrl, videoUrl);
    }
    getEmbedUrl(parsedUrl) {
        const formatter = this.getHostFormatter(parsedUrl.host);
        return formatter(parsedUrl);
    }
    getHostFormatter(host) {
        const hosts = {
            "www.youtube.com": (parsed) => this.editYoutubeUrl(parsed),
            "www.youtube-nocookie.com": (parsed) => this.editYoutubeUrl(parsed),
            "youtu.be": (parsed) => `https://www.youtube-nocookie.com/embed${parsed.path}`,
            "player.vimeo.com": (parsed) => parsed.url,
            "vimeo.com": (parsed) => `https://player.vimeo.com/video${parsed.path}`,
            "www.dailymotion.com": (parsed) => `https://www.dailymotion.com/embed${parsed.path}`,
        };
        return hosts[host];
    }
    editYoutubeUrl(parsedUrl) {
        const { path, searchObject } = parsedUrl;
        const { v: media } = searchObject;
        const params = buildParams(searchObject, ["v"]);
        if (path.includes("/embed")) {
            return `https://www.youtube-nocookie.com${path}${params}`;
        }
        if (media && typeof media === "string" && media.length > 1) {
            return `https://www.youtube-nocookie.com/embed/${media}${params}`;
        }
        snackbar(t("snackbars.wrong-url-format"), "error");
    }
    setIframeUrl(embedUrl) {
        const iframe = this.element?.querySelector("iframe");
        if (iframe) {
            iframe.src = embedUrl;
        }
        else {
            this.element.innerHTML = html `<iframe
        src="${embedUrl}"
        frameborder="0"
        webkitAllowFullScreen
        mozallowfullscreen
        allowfullscreen
      ></iframe>`;
            this.element.appendChild(this.mask);
        }
    }
}
