import Brick from "../core/brick.js";
import SpotifyModal from "../user-interface/plugin/spotify-modal.js";
import { snackbar } from "../user-interface/common/eden-snackbar.js";
import t from "../i18n/index.js";
const PLAYER_HOST = "https://open.spotify.com/embed/";
const DEFAULT_WIDTH = "300";
const DEFAULT_HEIGHT = "380";
const getType = (uri) => uri.split(":")[1];
const getCode = (uri) => uri.split(":")[2];
const buildUri = (url) => url.replace(PLAYER_HOST, "spotify:").replace("/", ":");
const applyIframeAttributes = (iframe, src, width, height) => {
    iframe.setAttribute("src", src);
    iframe.setAttribute("width", width);
    iframe.setAttribute("height", height);
    iframe.style.maxWidth = "100%";
    iframe.setAttribute("frameborder", "0");
    iframe.setAttribute("allowtransparency", "true");
    iframe.setAttribute("allow", "encrypted-media");
};
export default class Spotify extends Brick {
    constructor({ block, element, parentBricks }) {
        super({ block, element, parentBricks });
        this.handleAddPlaylistAction = () => {
            const iframe = this.getIframe();
            const url = iframe?.getAttribute("src") || "";
            const uri = buildUri(url);
            const { width, height } = this.getPlayerDimensions();
            new SpotifyModal(this.updateUrl, uri || "", width, height);
        };
        this.updateUrl = (uri, width, height) => {
            const type = getType(uri);
            const code = getCode(uri);
            if (!type || !code) {
                snackbar(t("snackbars.fill-valid-uri"), "error");
                return;
            }
            const newUrl = `${PLAYER_HOST}${type}/${code}`;
            const existingIframe = this.getIframe();
            if (existingIframe) {
                applyIframeAttributes(existingIframe, newUrl, width, height);
            }
            else {
                this.createAndInsertIframe(newUrl, width, height);
            }
        };
        this.activate = () => {
            if (!this.element)
                return;
            this.element.addEventListener("click", this.handleClick);
            if (this.getIframe()) {
                this.element.appendChild(this.mask);
            }
        };
        this.desactivate = () => {
            if (!this.element)
                return;
            this.element.removeEventListener("click", this.handleClick);
            this.mask.remove();
        };
        this.mask = this.createMask();
        this.actions = [
            {
                tooltip: t("bricks.spotify.add-playlist"),
                icon: "playlist",
                action: this.handleAddPlaylistAction,
            },
        ];
    }
    createMask() {
        const mask = document.createElement("div");
        mask.style.cssText =
            "position: absolute; top: 0; bottom: 0; left: 0; right: 0;";
        return mask;
    }
    getIframe() {
        return this.element.querySelector("iframe");
    }
    getPlayerDimensions() {
        const iframe = this.getIframe();
        return {
            width: (iframe && iframe.getAttribute("width")) || DEFAULT_WIDTH,
            height: (iframe && iframe.getAttribute("height")) || DEFAULT_HEIGHT,
        };
    }
    createAndInsertIframe(src, width, height) {
        const newIframe = document.createElement("iframe");
        applyIframeAttributes(newIframe, src, width, height);
        this.element.innerHTML = newIframe.outerHTML;
        this.element.appendChild(this.mask);
    }
}
