import Brick from "../core/brick.js";
import SlideshowModal from "../user-interface/plugin/slideshow-modal.js";
import { htmlStringToElement } from "../services/html.js";
import t from "../i18n/index.js";
export default class Slideshow extends Brick {
    constructor({ block, element, parentBricks, markup }) {
        super({ block, element, parentBricks });
        this.createSlideShow = (images, credits) => {
            this.element.innerHTML = "";
            this.renderImages(images);
            this.updateCredits(credits);
            this.desactivate();
            this.activate();
        };
        this.updateCredits = (text) => {
            if (text) {
                this.element.setAttribute("data-credits", text);
            }
            else {
                this.element.removeAttribute("data-credits");
            }
        };
        this.activate = () => {
            if (this.element) {
                this.element.addEventListener("click", this.handleClick);
            }
        };
        this.desactivate = () => {
            if (this.element) {
                this.element.removeEventListener("click", this.handleClick);
            }
        };
        this.markup = markup;
        this.actions = this.createActions();
    }
    createActions() {
        return [
            {
                tooltip: t("bricks.slideshow.select-images"),
                icon: "slideshow",
                action: () => this.handleSelectImagesAction(),
            },
        ];
    }
    handleSelectImagesAction() {
        const imgs = this.extractImagesData();
        const img = this.element.querySelector("img");
        if (!img)
            return;
        const credits = this.element.getAttribute("data-credits") || "";
        const { width, height } = this.extractImageDimensions(img);
        new SlideshowModal(this.createSlideShow, imgs, width, height, credits);
    }
    extractImagesData() {
        return Array.from(this.element.querySelectorAll("img")).map((img) => ({
            src: img.getAttribute("src") || "",
            alt: img.getAttribute("alt") || "",
            metadata: img.getAttribute("data-metadata"),
        }));
    }
    extractImageDimensions(img) {
        const width = img.hasAttribute("width")
            ? parseInt(img.getAttribute("width") || "0", 10)
            : 0;
        const height = img.hasAttribute("height")
            ? parseInt(img.getAttribute("height") || "0", 10)
            : 0;
        return { width, height };
    }
    renderImages(images) {
        images.forEach((image) => {
            if (!this.markup)
                return;
            const node = this.element.appendChild(htmlStringToElement(this.markup));
            const imgNode = node.nodeName === "IMG" ? node : node.querySelector("img");
            if (!imgNode)
                return;
            this.applyImageAttributes(imgNode, image);
        });
    }
    applyImageAttributes(imgNode, image) {
        imgNode.setAttribute("src", image.src);
        imgNode.setAttribute("alt", image.alt);
        if (image.metadata) {
            imgNode.setAttribute("data-metadata", image.metadata);
        }
    }
}
