import Brick from "../core/brick.js";
import t from "../i18n/index.js";
export default class Link extends Brick {
    constructor({ block, element, parentBricks, options, isCreation, }) {
        super({ block, element, parentBricks });
        this.initElement = (options) => {
            const { defaultHref, defaultTitle, defaultTarget } = options;
            if (defaultHref) {
                this.element.href = defaultHref;
            }
            if (defaultTitle) {
                this.element.title = defaultTitle;
            }
            if (defaultTarget) {
                this.element.target = defaultTarget;
                if (defaultTarget === "_blank") {
                    this.element.setAttribute("rel", "noopener");
                }
            }
        };
        this.updateLink = (url = "", title, target) => {
            this.setLinkHref(url);
            this.setLinkTitle(title);
            this.setLinkTarget(target ?? false);
        };
        this.activate = () => {
            if (!this.element)
                return;
            this.element.addEventListener("click", this.handleClick);
        };
        this.desactivate = () => {
            if (!this.element)
                return;
            this.element.removeEventListener("click", this.handleClick);
        };
        if (isCreation && options) {
            this.initElement(options);
        }
        this.actions = [this.createModifyLinkAction()];
    }
    createModifyLinkAction() {
        return {
            tooltip: t("bricks.link.modify-link"),
            icon: "link",
            action: async () => {
                const attributes = this.getAttributes();
                const { default: LinkModal } = await import("../user-interface/plugin/link-modal.js");
                new LinkModal(this.updateLink, attributes.href, attributes.title, attributes.isExternal).init();
            },
        };
    }
    getAttributes() {
        return {
            href: this.element.getAttribute("href") ?? "",
            title: this.element.title,
            isExternal: this.element.target === "_blank",
        };
    }
    setLinkHref(url) {
        if (url === "") {
            this.element.removeAttribute("href");
            return;
        }
        const patchedUrl = this.normalizeLinkUrl(url);
        this.element.setAttribute("href", patchedUrl);
    }
    normalizeLinkUrl(url) {
        const domain = this.getCurrentDomain();
        let normalized = url.replace(domain, "");
        if (normalized.startsWith("www.")) {
            normalized = `https://${normalized}`;
        }
        return normalized;
    }
    getCurrentDomain() {
        const { protocol, host } = window.location;
        return `${protocol}//${host}`;
    }
    setLinkTitle(title) {
        if (title && title !== "") {
            this.element.setAttribute("title", title);
        }
        else {
            this.element.removeAttribute("title");
        }
    }
    setLinkTarget(isExternal) {
        if (isExternal) {
            this.element.setAttribute("target", "_blank");
            this.element.setAttribute("rel", "noopener");
        }
        else {
            this.element.removeAttribute("target");
            this.element.removeAttribute("rel");
        }
    }
}
