import Brick from "../core/brick.js";
import { parseUrl } from "../services/url.js";
import IframeModal from "../user-interface/plugin/iframe-modal.js";
import { snackbar } from "../user-interface/common/eden-snackbar.js";
import t from "../i18n/index.js";
export default class Iframe extends Brick {
    constructor({ block, element, parentBricks }) {
        super({ block, element, parentBricks });
        this.updateIframe = (url, width, height, allowFullscreen) => {
            const parsedUrl = parseUrl(url);
            if (!parsedUrl?.host) {
                snackbar(t("snackbars.fill-valid-url"), "error");
                return;
            }
            let iframe = this.element?.querySelector("iframe");
            if (!iframe) {
                iframe = this.createIframe();
                this.element?.appendChild(iframe);
                this.element?.appendChild(this.mask);
            }
            this.setIframeAttributes(iframe, parsedUrl.url, width, height, allowFullscreen);
        };
        this.activate = () => {
            if (!this.element)
                return;
            this.element.addEventListener("click", this.handleClick);
            const iframe = this.element.querySelector("iframe");
            if (iframe) {
                this.element.style.position = "relative";
                this.element.appendChild(this.mask);
            }
        };
        this.desactivate = () => {
            if (!this.element)
                return;
            this.element.removeEventListener("click", this.handleClick);
            this.mask.remove();
            this.element.removeAttribute("style");
        };
        this.mask = this.createMask();
        this.actions = [this.createInsertContentAction()];
    }
    createMask() {
        const mask = document.createElement("div");
        mask.style.cssText =
            "position: absolute; top: 0; bottom: 0; left: 0; right: 0;";
        return mask;
    }
    createInsertContentAction() {
        return {
            tooltip: t("bricks.iframe.insert-content"),
            icon: "source",
            action: () => {
                const attributes = this.getIframeAttributes();
                new IframeModal(this.updateIframe, attributes.url, attributes.width, attributes.height, attributes.allowFullscreen);
            },
        };
    }
    getIframeAttributes() {
        const iframe = this.element?.querySelector("iframe");
        return {
            url: iframe?.getAttribute("src") ?? "",
            width: iframe?.getAttribute("width") ?? "1000",
            height: iframe?.getAttribute("height") ?? "450",
            allowFullscreen: iframe?.getAttribute("allow") === "fullscreen",
        };
    }
    createIframe() {
        const iframe = document.createElement("iframe");
        iframe.style.maxWidth = "100%";
        iframe.setAttribute("frameborder", "0");
        iframe.setAttribute("allowtransparency", "true");
        return iframe;
    }
    setIframeAttributes(iframe, url, width, height, allowFullscreen) {
        iframe.setAttribute("src", url);
        iframe.setAttribute("width", width);
        iframe.setAttribute("height", height);
        if (allowFullscreen) {
            iframe.setAttribute("allow", "fullscreen");
        }
        else {
            iframe.removeAttribute("allow");
        }
    }
}
