import { describe, it, expect, beforeEach } from "vitest";
import { toStatic, toLocalized, getRelative } from "../path";
describe("Path Utils", () => {
    describe("toStatic", () => {
        it("should add trailing slash to relative paths", () => {
            expect(toStatic("/about")).toBe("/about/");
            expect(toStatic("/contact")).toBe("/contact/");
        });
        it("should not add trailing slash if already present", () => {
            expect(toStatic("/about/")).toBe("/about/");
            expect(toStatic("/")).toBe("/");
        });
        it("should preserve anchors", () => {
            expect(toStatic("/about#section")).toBe("/about/#section");
            expect(toStatic("/page#top")).toBe("/page/#top");
        });
        it("should handle external URLs without adding slash", () => {
            expect(toStatic("https://example.com")).toBe("https://example.com");
            expect(toStatic("http://test.com/page")).toBe("http://test.com/page");
        });
        it("should not add slash to PDF files", () => {
            expect(toStatic("/document.pdf")).toBe("/document.pdf");
            expect(toStatic("/files/report.pdf")).toBe("/files/report.pdf");
        });
        it("should handle PDF with anchor", () => {
            expect(toStatic("/doc.pdf#page=2")).toBe("/doc.pdf#page=2");
        });
        it("should handle paths with trailing slash and anchor", () => {
            expect(toStatic("/about/#section")).toBe("/about/#section");
        });
        it("should handle empty or root paths", () => {
            expect(toStatic("/")).toBe("/");
        });
    });
    describe("toLocalized", () => {
        it("should add language prefix when langPrefix is true", () => {
            expect(toLocalized("/about", "fr", "true")).toBe("/fr/about");
            expect(toLocalized("/contact", "en", "true")).toBe("/en/contact");
        });
        it("should not add prefix when langPrefix is false", () => {
            expect(toLocalized("/about", "fr", "")).toBe("/about");
            expect(toLocalized("/contact", "en", null)).toBe("/contact");
        });
        it("should handle root path", () => {
            expect(toLocalized("/", "fr", "true")).toBe("/fr/");
            expect(toLocalized("/", "en", "")).toBe("/");
        });
        it("should handle paths with trailing slash", () => {
            expect(toLocalized("/about/", "fr", "true")).toBe("/fr/about/");
        });
        it("should handle complex paths", () => {
            expect(toLocalized("/blog/post/123", "de", "true")).toBe("/de/blog/post/123");
        });
    });
    describe("getRelative", () => {
        beforeEach(() => {
            Object.defineProperty(window, "location", {
                value: {
                    protocol: "https:",
                    host: "example.com",
                },
                writable: true,
            });
        });
        it("should return relative path from absolute URL", () => {
            expect(getRelative("https://example.com/about")).toBe("/about");
            expect(getRelative("https://example.com/contact")).toBe("/contact");
        });
        it("should return relative path as is", () => {
            expect(getRelative("/about")).toBe("/about");
            expect(getRelative("/contact/")).toBe("/contact/");
        });
        it("should return undefined for external URLs", () => {
            expect(getRelative("https://other.com/page")).toBeUndefined();
            expect(getRelative("http://external.com")).toBeUndefined();
        });
        it("should return undefined for protocol-relative URLs", () => {
            expect(getRelative("//cdn.example.com/file.js")).toBeUndefined();
        });
        it("should return undefined for URLs with dots in path", () => {
            expect(getRelative("/file.pdf")).toBeUndefined();
            expect(getRelative("/image.jpg")).toBeUndefined();
        });
        it("should return undefined for empty or invalid paths", () => {
            expect(getRelative("")).toBeUndefined();
            expect(getRelative(null)).toBeUndefined();
        });
        it("should handle root path", () => {
            expect(getRelative("/")).toBe("/");
        });
        it("should handle paths with query strings", () => {
            expect(getRelative("/search?q=test")).toBe("/search?q=test");
        });
        it("should handle paths with anchors", () => {
            expect(getRelative("/page#section")).toBe("/page#section");
        });
        it("should remove host from URL", () => {
            expect(getRelative("https://example.com/test/path")).toBe("/test/path");
        });
    });
});
