import { describe, it, expect, vi, beforeEach } from "vitest";
import { getJson, getText, GET, POST } from "../http";
describe("HTTP Utils", () => {
    beforeEach(() => {
        vi.clearAllMocks();
        global.fetch = vi.fn();
    });
    describe("getJson", () => {
        it("should fetch and parse JSON", async () => {
            const mockData = { test: "data" };
            const mockResponse = {
                json: vi.fn().mockResolvedValue(mockData),
            };
            vi.mocked(global.fetch).mockResolvedValue(mockResponse);
            const result = await getJson("/test.json");
            expect(global.fetch).toHaveBeenCalledWith("/test.json");
            expect(result).toEqual(mockData);
        });
        it("should retry on failure", async () => {
            const mockData = { test: "data" };
            const mockResponse = {
                json: vi.fn().mockResolvedValue(mockData),
            };
            vi.mocked(global.fetch)
                .mockRejectedValueOnce(new Error("Network error"))
                .mockRejectedValueOnce(new Error("Network error"))
                .mockResolvedValue(mockResponse);
            const consoleSpy = vi.spyOn(console, "warn").mockImplementation(() => { });
            const result = await getJson("/test.json");
            expect(global.fetch).toHaveBeenCalledTimes(3);
            expect(consoleSpy).toHaveBeenCalledTimes(2);
            expect(result).toEqual(mockData);
            consoleSpy.mockRestore();
        });
        it("should throw error after max retries", async () => {
            vi.mocked(global.fetch).mockRejectedValue(new Error("Network error"));
            const consoleSpy = vi.spyOn(console, "warn").mockImplementation(() => { });
            await expect(getJson("/test.json")).rejects.toThrow("Network error");
            expect(global.fetch).toHaveBeenCalledTimes(6);
            consoleSpy.mockRestore();
        });
        it("should use custom retry count", async () => {
            vi.mocked(global.fetch).mockRejectedValue(new Error("Network error"));
            const consoleSpy = vi.spyOn(console, "warn").mockImplementation(() => { });
            await expect(getJson("/test.json", 2)).rejects.toThrow("Network error");
            expect(global.fetch).toHaveBeenCalledTimes(3);
            consoleSpy.mockRestore();
        });
    });
    describe("getText", () => {
        it("should fetch and return text", async () => {
            const mockText = "Hello World";
            const mockResponse = {
                text: vi.fn().mockResolvedValue(mockText),
            };
            vi.mocked(global.fetch).mockResolvedValue(mockResponse);
            const result = await getText("https://api.test.com", "/test", {});
            expect(global.fetch).toHaveBeenCalledWith("https://api.test.com/api/test?");
            expect(result).toBe(mockText);
        });
        it("should append query parameters", async () => {
            const mockResponse = {
                text: vi.fn().mockResolvedValue("text"),
            };
            vi.mocked(global.fetch).mockResolvedValue(mockResponse);
            await getText("https://api.test.com", "/test", {
                foo: "bar",
                baz: "qux",
            });
            expect(global.fetch).toHaveBeenCalledWith("https://api.test.com/api/test?foo=bar&baz=qux");
        });
        it("should retry on failure", async () => {
            const mockResponse = {
                text: vi.fn().mockResolvedValue("text"),
            };
            vi.mocked(global.fetch)
                .mockRejectedValueOnce(new Error("Network error"))
                .mockResolvedValue(mockResponse);
            const consoleSpy = vi.spyOn(console, "warn").mockImplementation(() => { });
            const result = await getText("https://api.test.com", "/test", {});
            expect(global.fetch).toHaveBeenCalledTimes(2);
            expect(result).toBe("text");
            consoleSpy.mockRestore();
        });
        it("should throw error after max retries", async () => {
            vi.mocked(global.fetch).mockRejectedValue(new Error("Network error"));
            const consoleSpy = vi.spyOn(console, "warn").mockImplementation(() => { });
            await expect(getText("https://api.test.com", "/test", {})).rejects.toThrow("Network error");
            consoleSpy.mockRestore();
        });
    });
    describe("GET", () => {
        it("should make GET request and parse JSON", async () => {
            const mockData = { result: "success" };
            const mockResponse = {
                json: vi.fn().mockResolvedValue(mockData),
            };
            vi.mocked(global.fetch).mockResolvedValue(mockResponse);
            const result = await GET("https://api.test.com", "/endpoint", {});
            expect(global.fetch).toHaveBeenCalledWith("https://api.test.com/api/endpoint?");
            expect(result).toEqual(mockData);
        });
        it("should append query parameters to GET request", async () => {
            const mockResponse = {
                json: vi.fn().mockResolvedValue({}),
            };
            vi.mocked(global.fetch).mockResolvedValue(mockResponse);
            await GET("https://api.test.com", "/endpoint", {
                param1: "value1",
                param2: "value2",
            });
            expect(global.fetch).toHaveBeenCalledWith("https://api.test.com/api/endpoint?param1=value1&param2=value2");
        });
    });
    describe("POST", () => {
        it("should make POST request with body", async () => {
            const mockData = { result: "success" };
            const mockResponse = {
                json: vi.fn().mockResolvedValue(mockData),
            };
            vi.mocked(global.fetch).mockResolvedValue(mockResponse);
            const body = { key: "value" };
            const result = await POST("https://api.test.com", "/endpoint", body);
            expect(global.fetch).toHaveBeenCalledWith("https://api.test.com/api/endpoint", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify(body),
            });
            expect(result).toEqual(mockData);
        });
        it("should include custom headers", async () => {
            const mockResponse = {
                json: vi.fn().mockResolvedValue({}),
            };
            vi.mocked(global.fetch).mockResolvedValue(mockResponse);
            await POST("https://api.test.com", "/endpoint", {}, { authorization: "Bearer token" });
            expect(global.fetch).toHaveBeenCalledWith("https://api.test.com/api/endpoint", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    authorization: "Bearer token",
                },
                body: "{}",
            });
        });
    });
});
