import { describe, it, expect } from "vitest";
import * as Menu from "../menu";
describe("Menu Service", () => {
    describe("isMenuSelected", () => {
        it("should return false when menuPath is null", () => {
            const item = {
                path: "/test",
                title: "Test",
                name: "test",
                menu: "main",
            };
            const result = Menu.isMenuSelected(undefined, item);
            expect(result).toBe(false);
        });
        it("should return false when item is null", () => {
            const menuPath = {
                id: 1,
                active: false,
                disabled: false,
                name: "test",
                title: "Test",
                menu: "main",
                path: "/test",
                next: null,
                prev: null,
            };
            const result = Menu.isMenuSelected(menuPath, undefined);
            expect(result).toBe(false);
        });
        it("should return false when both parameters are null", () => {
            const result = Menu.isMenuSelected(undefined, undefined);
            expect(result).toBe(false);
        });
        it("should return true when paths match exactly", () => {
            const menuPath = {
                id: 1,
                active: false,
                disabled: false,
                name: "test",
                title: "Test",
                menu: "main",
                path: "/test",
                next: null,
                prev: null,
            };
            const item = {
                path: "/test",
                title: "Test",
                name: "test",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(true);
        });
        it("should return false when paths do not match", () => {
            const menuPath = {
                id: 1,
                active: false,
                disabled: false,
                name: "test",
                title: "Test",
                menu: "main",
                path: "/test",
                next: null,
                prev: null,
            };
            const item = {
                path: "/other",
                title: "Other",
                name: "other",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(false);
        });
        it("should check previous items in the chain", () => {
            const prev = {
                id: 1,
                active: false,
                disabled: false,
                name: "home",
                title: "Home",
                menu: "main",
                path: "/",
                next: null,
                prev: null,
            };
            const menuPath = {
                id: 2,
                active: false,
                disabled: false,
                name: "about",
                title: "About",
                menu: "main",
                path: "/about",
                next: null,
                prev: prev,
            };
            prev.next = menuPath;
            const item = {
                path: "/",
                title: "Home",
                name: "home",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(true);
        });
        it("should return false when item not in chain", () => {
            const prev = {
                id: 1,
                active: false,
                disabled: false,
                name: "home",
                title: "Home",
                menu: "main",
                path: "/",
                next: null,
                prev: null,
            };
            const menuPath = {
                id: 2,
                active: false,
                disabled: false,
                name: "about",
                title: "About",
                menu: "main",
                path: "/about",
                next: null,
                prev: prev,
            };
            prev.next = menuPath;
            const item = {
                path: "/contact",
                title: "Contact",
                name: "contact",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(false);
        });
        it("should handle long chain of menu items", () => {
            const home = {
                id: 1,
                active: false,
                disabled: false,
                name: "home",
                title: "Home",
                menu: "main",
                path: "/",
                next: null,
                prev: null,
            };
            const about = {
                id: 2,
                active: false,
                disabled: false,
                name: "about",
                title: "About",
                menu: "main",
                path: "/about",
                next: null,
                prev: home,
            };
            const team = {
                id: 3,
                active: false,
                disabled: false,
                name: "team",
                title: "Team",
                menu: "main",
                path: "/about/team",
                next: null,
                prev: about,
            };
            const member = {
                id: 4,
                active: false,
                disabled: false,
                name: "member",
                title: "Member",
                menu: "main",
                path: "/about/team/member",
                next: null,
                prev: team,
            };
            home.next = about;
            about.next = team;
            team.next = member;
            const homeItem = {
                path: "/",
                title: "Home",
                name: "home",
                menu: "main",
            };
            const result = Menu.isMenuSelected(member, homeItem);
            expect(result).toBe(true);
        });
        it("should handle chain where item is at the end", () => {
            const prev = {
                id: 1,
                active: false,
                disabled: false,
                name: "home",
                title: "Home",
                menu: "main",
                path: "/",
                next: null,
                prev: null,
            };
            const menuPath = {
                id: 2,
                active: false,
                disabled: false,
                name: "about",
                title: "About",
                menu: "main",
                path: "/about",
                next: null,
                prev: prev,
            };
            prev.next = menuPath;
            const item = {
                path: "/about",
                title: "About",
                name: "about",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(true);
        });
        it("should be case-sensitive for path matching", () => {
            const menuPath = {
                id: 1,
                active: false,
                disabled: false,
                name: "test",
                title: "Test",
                menu: "main",
                path: "/Test",
                next: null,
                prev: null,
            };
            const item = {
                path: "/test",
                title: "Test",
                name: "test",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(false);
        });
        it("should match root path correctly", () => {
            const menuPath = {
                id: 1,
                active: false,
                disabled: false,
                name: "home",
                title: "Home",
                menu: "main",
                path: "/",
                next: null,
                prev: null,
            };
            const item = {
                path: "/",
                title: "Home",
                name: "home",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(true);
        });
        it("should handle deeply nested paths", () => {
            const menuPath = {
                id: 1,
                active: false,
                disabled: false,
                name: "deep",
                title: "Deep",
                menu: "main",
                path: "/a/b/c/d/e/f",
                next: null,
                prev: null,
            };
            const item = {
                path: "/a/b/c/d/e/f",
                title: "Deep",
                name: "deep",
                menu: "main",
            };
            const result = Menu.isMenuSelected(menuPath, item);
            expect(result).toBe(true);
        });
    });
});
