import { describe, it, expect, vi, beforeEach } from "vitest";
import * as Eden from "../eden";
import * as API from "../api";
vi.mock("../api", () => ({
    getConfig: vi.fn(),
}));
describe("Eden Service", () => {
    beforeEach(() => {
        vi.clearAllMocks();
        if (typeof sessionStorage !== "undefined") {
            sessionStorage.clear();
        }
    });
    describe("EDEN_TOKEN constant", () => {
        it("should have correct token key", () => {
            expect(Eden.EDEN_TOKEN).toBe("eden_token");
        });
    });
    describe("isConnected", () => {
        it("should return false when sessionStorage is not available", () => {
            const mockSessionStorage = global.sessionStorage;
            delete global.sessionStorage;
            const result = Eden.isConnected();
            expect(result).toBe(false);
            global.sessionStorage = mockSessionStorage;
        });
        it("should return false when token is not in sessionStorage", () => {
            if (typeof sessionStorage !== "undefined") {
                sessionStorage.clear();
            }
            const result = Eden.isConnected();
            expect(result).toBe(false);
        });
        it("should return false when token format is invalid", () => {
            if (typeof sessionStorage !== "undefined") {
                sessionStorage.setItem(Eden.EDEN_TOKEN, "invalid-format");
            }
            const result = Eden.isConnected();
            expect(result).toBe(false);
        });
        it("should return false when token has invalid base64", () => {
            if (typeof sessionStorage !== "undefined") {
                sessionStorage.setItem(Eden.EDEN_TOKEN, "header.!!!invalid!!!.signature");
            }
            const result = Eden.isConnected();
            expect(result).toBe(false);
        });
        it("should return true when valid token exists", () => {
            if (typeof sessionStorage !== "undefined") {
                const payload = JSON.stringify({ sub: "user123", iat: 1234567890 });
                const encoded = btoa(unescape(encodeURIComponent(payload)));
                const token = `header.${encoded}.signature`;
                sessionStorage.setItem(Eden.EDEN_TOKEN, token);
            }
            const result = Eden.isConnected();
            expect(result).toBe(true);
        });
        it("should return true when token has complex payload", () => {
            if (typeof sessionStorage !== "undefined") {
                const payload = JSON.stringify({
                    sub: "user123",
                    iat: 1234567890,
                    exp: 1234571490,
                    role: "admin",
                    email: "test@example.com",
                });
                const encoded = btoa(unescape(encodeURIComponent(payload)));
                const token = `header.${encoded}.signature`;
                sessionStorage.setItem(Eden.EDEN_TOKEN, token);
            }
            const result = Eden.isConnected();
            expect(result).toBe(true);
        });
        it("should handle special characters in token", () => {
            if (typeof sessionStorage !== "undefined") {
                const payload = JSON.stringify({
                    sub: "user@123",
                    iat: 1234567890,
                    email: "user+test@example.com",
                });
                const encoded = btoa(unescape(encodeURIComponent(payload)));
                const token = `header.${encoded}.signature`;
                sessionStorage.setItem(Eden.EDEN_TOKEN, token);
            }
            const result = Eden.isConnected();
            expect(result).toBe(true);
        });
    });
    describe("setup", () => {
        it("should fetch config and set window.EdenConfig", async () => {
            const mockEdenConfig = {
                edenApi: "https://api.test.com",
                site: "test-site",
                siteName: "Test Site",
                lang: "en",
                langs: ["en", "fr"],
                langPrefix: true,
                domains: { en: "example.com", fr: "example.fr" },
                layout: "default",
                layouts: [],
                preProd: false,
            };
            const mockWindowConfig = {};
            vi.mocked(API.getConfig).mockResolvedValue(mockEdenConfig);
            await Eden.setup(mockWindowConfig);
            expect(API.getConfig).toHaveBeenCalled();
            expect(window.EdenConfig).toEqual({
                ...mockEdenConfig,
                ...mockWindowConfig,
            });
        });
        it("should merge user config with fetched config", async () => {
            const mockEdenConfig = {
                edenApi: "https://api.test.com",
                site: "test-site",
                siteName: "Test",
                lang: "en",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                layout: "default",
                layouts: [],
            };
            const userConfig = {};
            vi.mocked(API.getConfig).mockResolvedValue(mockEdenConfig);
            await Eden.setup(userConfig);
            expect(window.EdenConfig).toHaveProperty("edenApi", "https://api.test.com");
        });
        it("should allow user config to override fetched config", async () => {
            const mockEdenConfig = {
                edenApi: "https://api.test.com",
                site: "test-site",
                siteName: "Test",
                lang: "en",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                layout: "default",
                layouts: [],
            };
            const userConfig = { site: "override-site" };
            vi.mocked(API.getConfig).mockResolvedValue(mockEdenConfig);
            await Eden.setup(userConfig);
            expect(window.EdenConfig.site).toBe("override-site");
        });
        it("should handle empty user config", async () => {
            const mockEdenConfig = {
                edenApi: "https://api.test.com",
                site: "test-site",
                siteName: "Test",
                lang: "en",
                langs: ["en"],
                langPrefix: false,
                domains: { en: "example.com" },
                layout: "default",
                layouts: [],
            };
            vi.mocked(API.getConfig).mockResolvedValue(mockEdenConfig);
            await Eden.setup({});
            expect(window.EdenConfig).toEqual(mockEdenConfig);
        });
        it("should handle config with multiple properties", async () => {
            const mockEdenConfig = {
                edenApi: "https://api.test.com",
                site: "test-site",
                siteName: "Test Site",
                lang: "en",
                langs: ["en", "fr", "de"],
                langPrefix: true,
                domains: {
                    en: "example.com",
                    fr: "example.fr",
                    de: "example.de",
                },
                layout: "default",
                layouts: [],
                preProd: ["en"],
            };
            const userConfig = {};
            vi.mocked(API.getConfig).mockResolvedValue(mockEdenConfig);
            await Eden.setup(userConfig);
            expect(Object.keys(window.EdenConfig).length).toBeGreaterThan(Object.keys(mockEdenConfig).length - 1);
        });
    });
});
