import { describe, it, expect, beforeEach, afterEach } from "vitest";
import * as ConfigService from "../config";
describe("Config Service", () => {
    const mockConfig = {
        edenApi: "https://api.example.com",
        site: "test-site",
        langs: ["en", "fr"],
        langPrefix: true,
        domains: { en: "example.com", fr: "example.fr" },
        siteName: "Test Site",
        preProd: false,
    };
    describe("buildConfig", () => {
        it("should build config with all properties", () => {
            const result = ConfigService.default(mockConfig);
            expect(result.trailingSlash).toBe(true);
            expect(result.env.EDEN_API).toBe("https://api.example.com");
            expect(result.env.SITE).toBe("test-site");
            expect(result.env.LANGS).toBe("en,fr");
            expect(result.env.LANG_PREFIX).toBe("true");
            expect(result.env.SITE_NAME).toBe("Test Site");
        });
        it("should stringify complex objects in env", () => {
            const result = ConfigService.default(mockConfig);
            expect(result.env.DOMAINS).toBe(JSON.stringify({ en: "example.com", fr: "example.fr" }));
            expect(result.env.PRE_PROD).toBe("false");
            expect(result.env.LANG_PREFIX).toBe("true");
        });
        it("should handle single language", () => {
            const singleLangConfig = {
                ...mockConfig,
                langs: ["en"],
            };
            const result = ConfigService.default(singleLangConfig);
            expect(result.env.LANGS).toBe("en");
        });
        it("should handle array preProd", () => {
            const configWithArrayPreprod = {
                ...mockConfig,
                preProd: ["en"],
            };
            const result = ConfigService.default(configWithArrayPreprod);
            expect(result.env.PRE_PROD).toBe(JSON.stringify(["en"]));
        });
        it("should handle langPrefix as false", () => {
            const configNoPrefix = {
                ...mockConfig,
                langPrefix: false,
            };
            const result = ConfigService.default(configNoPrefix);
            expect(result.env.LANG_PREFIX).toBe("false");
        });
        it("should handle multiple languages", () => {
            const multiLangConfig = {
                ...mockConfig,
                langs: ["en", "fr", "de", "es"],
            };
            const result = ConfigService.default(multiLangConfig);
            expect(result.env.LANGS).toBe("en,fr,de,es");
        });
        it("should handle complex domains object", () => {
            const configWithMultipleDomains = {
                ...mockConfig,
                domains: {
                    en: "example.com",
                    fr: "example.fr",
                    de: "example.de",
                    es: "example.es",
                },
            };
            const result = ConfigService.default(configWithMultipleDomains);
            const domainsObj = JSON.parse(result.env.DOMAINS);
            expect(domainsObj.en).toBe("example.com");
            expect(domainsObj.fr).toBe("example.fr");
            expect(domainsObj.de).toBe("example.de");
            expect(domainsObj.es).toBe("example.es");
        });
    });
    describe("getConfig", () => {
        let originalEnv;
        beforeEach(() => {
            originalEnv = process.env;
            process.env = {
                NODE_ENV: process.env.NODE_ENV || "test",
                EDEN_API: "https://api.example.com",
                SITE: "test-site",
                LANGS: "en,fr",
                LANG_PREFIX: "true",
                DOMAINS: JSON.stringify({ en: "example.com", fr: "example.fr" }),
                SITE_NAME: "Test Site",
                PRE_PROD: "false",
            };
        });
        afterEach(() => {
            process.env = originalEnv;
        });
        it("should read config from environment variables", () => {
            const result = ConfigService.getConfig();
            expect(result.edenApi).toBe("https://api.example.com");
            expect(result.site).toBe("test-site");
            expect(result.langs).toEqual(["en", "fr"]);
            expect(result.langPrefix).toBe(true);
            expect(result.domains).toEqual({
                en: "example.com",
                fr: "example.fr",
            });
            expect(result.siteName).toBe("Test Site");
            expect(result.preProd).toBe(false);
        });
        it("should parse langs as array", () => {
            process.env.LANGS = "en,fr,de";
            const result = ConfigService.getConfig();
            expect(Array.isArray(result.langs)).toBe(true);
            expect(result.langs).toEqual(["en", "fr", "de"]);
        });
        it("should parse langPrefix as boolean true", () => {
            process.env.LANG_PREFIX = "true";
            const result = ConfigService.getConfig();
            expect(result.langPrefix).toBe(true);
        });
        it("should parse langPrefix as boolean false", () => {
            process.env.LANG_PREFIX = "false";
            const result = ConfigService.getConfig();
            expect(result.langPrefix).toBe(false);
        });
        it("should parse domains as object", () => {
            process.env.DOMAINS = JSON.stringify({
                en: "example.com",
                fr: "example.fr",
                de: "example.de",
            });
            const result = ConfigService.getConfig();
            expect(result.domains.en).toBe("example.com");
            expect(result.domains.fr).toBe("example.fr");
            expect(result.domains.de).toBe("example.de");
        });
        it("should parse preProd as boolean", () => {
            process.env.PRE_PROD = "true";
            const result = ConfigService.getConfig();
            expect(result.preProd).toBe(true);
        });
        it("should parse preProd as array", () => {
            process.env.PRE_PROD = JSON.stringify(["en", "fr"]);
            const result = ConfigService.getConfig();
            expect(Array.isArray(result.preProd)).toBe(true);
            expect(result.preProd).toEqual(["en", "fr"]);
        });
        it("should handle single language environment", () => {
            process.env.LANGS = "en";
            const result = ConfigService.getConfig();
            expect(result.langs).toEqual(["en"]);
        });
        it("should handle empty domains", () => {
            process.env.DOMAINS = JSON.stringify({});
            const result = ConfigService.getConfig();
            expect(result.domains).toEqual({});
        });
        it("should maintain string values for edenApi, site, and siteName", () => {
            process.env.EDEN_API = "https://custom-api.com";
            process.env.SITE = "custom-site";
            process.env.SITE_NAME = "Custom Site";
            const result = ConfigService.getConfig();
            expect(typeof result.edenApi).toBe("string");
            expect(typeof result.site).toBe("string");
            expect(typeof result.siteName).toBe("string");
            expect(result.edenApi).toBe("https://custom-api.com");
            expect(result.site).toBe("custom-site");
            expect(result.siteName).toBe("Custom Site");
        });
    });
});
