import { describe, it, expect, vi, beforeEach } from "vitest";
import * as API from "../api";
import * as HTTP from "../../utils/http";
vi.mock("../../utils/http", () => ({
    GET: vi.fn(),
    POST: vi.fn(),
    getJson: vi.fn(),
}));
describe("API Service", () => {
    beforeEach(() => {
        vi.clearAllMocks();
    });
    describe("checkToken", () => {
        it("should make POST request to check token", async () => {
            const mockResponse = { valid: true };
            vi.mocked(HTTP.POST).mockResolvedValue(mockResponse);
            const result = await API.checkToken("https://api.test.com", "test-token");
            expect(HTTP.POST).toHaveBeenCalledWith("https://api.test.com", "/admin/login/check", {}, { authorization: "test-token" });
            expect(result).toEqual(mockResponse);
        });
    });
    describe("login", () => {
        it("should make POST request with credentials", async () => {
            const mockResponse = { token: "new-token" };
            vi.mocked(HTTP.POST).mockResolvedValue(mockResponse);
            const result = await API.login("https://api.test.com", "test-site", "user@test.com", "password123");
            expect(HTTP.POST).toHaveBeenCalledWith("https://api.test.com", "/admin/login", {
                site: "test-site",
                email: "user@test.com",
                password: "password123",
            });
            expect(result).toEqual(mockResponse);
        });
    });
    describe("getPage", () => {
        it("should make GET request for page data", async () => {
            const mockPage = {
                zones: { header: "content" },
                seo: { title: "Test" },
            };
            vi.mocked(HTTP.GET).mockResolvedValue(mockPage);
            const result = await API.getPage("https://api.test.com", "test-site", "en", "/test-path");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/page", {
                site: "test-site",
                lang: "en",
                path: "/test-path",
                skipReverse: "false",
            });
            expect(result).toEqual(mockPage);
        });
        it("should pass skipReverse parameter", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue({});
            await API.getPage("https://api.test.com", "test-site", "en", "/test", true);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/page", {
                site: "test-site",
                lang: "en",
                path: "/test",
                skipReverse: "true",
            });
        });
    });
    describe("getSeo", () => {
        it("should make GET request for SEO data", async () => {
            const mockSeo = {
                title: "Test Title",
                description: "Test Description",
            };
            vi.mocked(HTTP.GET).mockResolvedValue(mockSeo);
            const result = await API.getSeo("https://api.test.com", "test-site", "en", "/test");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/seo", {
                site: "test-site",
                lang: "en",
                path: "/test",
            });
            expect(result).toEqual(mockSeo);
        });
    });
    describe("getLocalizedPath", () => {
        it("should make GET request for localized path", async () => {
            const mockResponse = { fr: { path: "/fr/test" } };
            vi.mocked(HTTP.GET).mockResolvedValue(mockResponse);
            const result = await API.getLocalizedPath("https://api.test.com", "test-site", ["en", "fr"], "en", "/test", "fr");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/lang-switches", {
                site: "test-site",
                langs: "en,fr",
                currentLang: "en",
                path: "/test",
            });
            expect(result).toBe("/fr/test");
        });
        it("should return null if target language not found", async () => {
            const mockResponse = { fr: { path: "/fr/test" } };
            vi.mocked(HTTP.GET).mockResolvedValue(mockResponse);
            const result = await API.getLocalizedPath("https://api.test.com", "test-site", ["en", "fr"], "en", "/test", "de");
            expect(result).toBeNull();
        });
    });
    describe("getLangSwitches", () => {
        it("should make GET request for lang switches", async () => {
            const mockSwitches = {
                en: "/en/test",
                fr: "/fr/test",
            };
            vi.mocked(HTTP.GET).mockResolvedValue(mockSwitches);
            const result = await API.getLangSwitches("https://api.test.com", "test-site", "en", ["en", "fr"], "/test");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/lang-switches", {
                site: "test-site",
                currentLang: "en",
                langs: "en,fr",
                path: "/test",
            });
            expect(result).toEqual(mockSwitches);
        });
        it("should return empty object for single language", async () => {
            const result = await API.getLangSwitches("https://api.test.com", "test-site", "en", ["en"], "/test");
            expect(HTTP.GET).not.toHaveBeenCalled();
            expect(result).toEqual({});
        });
    });
    describe("getExportPathMap", () => {
        it("should make GET request for export path map", async () => {
            const mockPathMap = { "/": "index", "/about": "about" };
            vi.mocked(HTTP.GET).mockResolvedValue(mockPathMap);
            const result = await API.getExportPathMap("https://api.test.com", "test-site", "en");
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/export-path-map", {
                site: "test-site",
                langs: "en",
                ignoreMounted: "true",
            });
            expect(result).toEqual(mockPathMap);
        });
        it("should pass ignoreMounted parameter", async () => {
            vi.mocked(HTTP.GET).mockResolvedValue({});
            await API.getExportPathMap("https://api.test.com", "test-site", "en", false);
            expect(HTTP.GET).toHaveBeenCalledWith("https://api.test.com", "/export-path-map", {
                site: "test-site",
                langs: "en",
                ignoreMounted: "false",
            });
        });
    });
    describe("getConfig", () => {
        it("should fetch config from static file", async () => {
            const mockConfig = { site: "test", langs: ["en"] };
            vi.mocked(HTTP.getJson).mockResolvedValue(mockConfig);
            const result = await API.getConfig();
            expect(HTTP.getJson).toHaveBeenCalledWith("/static/cms/config.json");
            expect(result).toEqual(mockConfig);
        });
    });
    describe("updateDraftMode", () => {
        it("should call draft API with status and token", async () => {
            const mockResponse = { success: true };
            vi.mocked(HTTP.getJson).mockResolvedValue(mockResponse);
            const result = await API.updateDraftMode(true, "test-token");
            expect(HTTP.getJson).toHaveBeenCalledWith("/api/draft?status=true&token=test-token");
            expect(result).toEqual(mockResponse);
        });
        it("should handle false status", async () => {
            vi.mocked(HTTP.getJson).mockResolvedValue({});
            await API.updateDraftMode(false, "test-token");
            expect(HTTP.getJson).toHaveBeenCalledWith("/api/draft?status=false&token=test-token");
        });
    });
});
