import { describe, it, expect, vi, beforeEach } from "vitest";
import { renderHook, waitFor, fireEvent } from "@testing-library/react";
import usePatchedInternalLinks from "../use-patched-internal-links";
import * as Context from "../../components/context";
import * as Path from "../../utils/path";
const mockRouterPush = vi.fn();
const mockRouterReplace = vi.fn();
const mockRouterPrefetch = vi.fn();
vi.mock("next/navigation", () => ({
    useRouter: () => ({
        push: mockRouterPush,
        replace: mockRouterReplace,
        prefetch: mockRouterPrefetch,
    }),
}));
vi.mock("../../components/context", async () => {
    const actual = await vi.importActual("../../components/context");
    return {
        ...actual,
        useEdenContext: vi.fn(),
    };
});
vi.mock("../../utils/path", async () => {
    const actual = await vi.importActual("../../utils/path");
    return {
        ...actual,
        getRelative: vi.fn(),
    };
});
describe("usePatchedInternalLinks", () => {
    const mockContext = {
        lang: "en",
        path: "/test",
        isConnected: false,
        edenMode: null,
        config: {},
        pageProps: {},
        zones: {},
        edenApi: "",
        site: "",
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    beforeEach(() => {
        vi.clearAllMocks();
        mockRouterPush.mockClear();
        mockRouterReplace.mockClear();
        mockRouterPrefetch.mockClear();
        document.body.innerHTML = "";
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
    });
    it("should patch internal link clicks", async () => {
        vi.mocked(Path.getRelative).mockReturnValue("/internal");
        const section = document.createElement("section");
        section.dataset.template = "default";
        const link = document.createElement("a");
        link.href = "https://example.com/internal";
        section.appendChild(link);
        document.body.appendChild(section);
        renderHook(() => usePatchedInternalLinks());
        fireEvent.click(link);
        await waitFor(() => {
            expect(mockRouterPush).toHaveBeenCalledWith("/internal");
        });
    });
    it("should not patch external links", async () => {
        vi.mocked(Path.getRelative).mockReturnValue(undefined);
        const section = document.createElement("section");
        section.dataset.template = "default";
        const link = document.createElement("a");
        link.href = "https://external.com/page";
        section.appendChild(link);
        document.body.appendChild(section);
        renderHook(() => usePatchedInternalLinks());
        fireEvent.click(link);
        expect(mockRouterPush).not.toHaveBeenCalled();
    });
    it("should not patch links with target=_blank", async () => {
        vi.mocked(Path.getRelative).mockReturnValue("/internal");
        const section = document.createElement("section");
        section.dataset.template = "default";
        const link = document.createElement("a");
        link.href = "https://example.com/internal";
        link.target = "_blank";
        section.appendChild(link);
        document.body.appendChild(section);
        renderHook(() => usePatchedInternalLinks());
        fireEvent.click(link);
        expect(mockRouterPush).not.toHaveBeenCalled();
    });
    it("should not patch links outside template sections", () => {
        vi.mocked(Path.getRelative).mockReturnValue("/internal");
        const link = document.createElement("a");
        link.href = "https://example.com/internal";
        document.body.appendChild(link);
        renderHook(() => usePatchedInternalLinks());
        fireEvent.click(link);
        expect(mockRouterPush).not.toHaveBeenCalled();
    });
    it("should not patch when connected", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            isConnected: true,
        });
        vi.mocked(Path.getRelative).mockReturnValue("/internal");
        const section = document.createElement("section");
        section.dataset.template = "default";
        const link = document.createElement("a");
        link.href = "https://example.com/internal";
        section.appendChild(link);
        document.body.appendChild(section);
        renderHook(() => usePatchedInternalLinks());
        fireEvent.click(link);
        expect(mockRouterPush).not.toHaveBeenCalled();
    });
    it("should handle links without href", () => {
        const section = document.createElement("section");
        section.dataset.template = "default";
        const link = document.createElement("a");
        section.appendChild(link);
        document.body.appendChild(section);
        renderHook(() => usePatchedInternalLinks());
        expect(() => fireEvent.click(link)).not.toThrow();
        expect(mockRouterPush).not.toHaveBeenCalled();
    });
    it("should respect preventDefault", () => {
        vi.mocked(Path.getRelative).mockReturnValue("/internal");
        const section = document.createElement("section");
        section.dataset.template = "default";
        const link = document.createElement("a");
        link.href = "https://example.com/internal";
        link.addEventListener("click", (e) => e.preventDefault(), true);
        section.appendChild(link);
        document.body.appendChild(section);
        renderHook(() => usePatchedInternalLinks());
        fireEvent.click(link);
        expect(mockRouterPush).not.toHaveBeenCalled();
    });
    it("should cleanup event listener on unmount", () => {
        const removeEventListenerSpy = vi.spyOn(document, "removeEventListener");
        const { unmount } = renderHook(() => usePatchedInternalLinks());
        unmount();
        expect(removeEventListenerSpy).toHaveBeenCalledWith("click", expect.any(Function));
    });
    it("should handle clicks on nested elements", async () => {
        vi.mocked(Path.getRelative).mockReturnValue("/internal");
        const section = document.createElement("section");
        section.dataset.template = "default";
        const link = document.createElement("a");
        link.href = "https://example.com/internal";
        const span = document.createElement("span");
        span.textContent = "Click me";
        link.appendChild(span);
        section.appendChild(link);
        document.body.appendChild(section);
        renderHook(() => usePatchedInternalLinks());
        fireEvent.click(span);
        await waitFor(() => {
            expect(mockRouterPush).toHaveBeenCalledWith("/internal");
        });
    });
});
