import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, screen, waitFor } from "@testing-library/react";
import userEvent from "@testing-library/user-event";
import ResetConfirm from "../reset-confirm";
import * as Context from "../../context";
import * as CMS from "../../../services/eden";
import * as API from "../../../services/api";
vi.mock("../../context", () => ({
    useEdenContext: vi.fn(),
}));
vi.mock("../../../services/eden", () => ({
    EDEN_TOKEN: "eden-token",
}));
vi.mock("../../../services/api", () => ({
    confirmPasswordReset: vi.fn(),
}));
vi.mock("../../logo", () => ({
    default: () => _jsx("div", { "data-testid": "logo", children: "Logo" }),
}));
vi.mock("../../icons/eye", () => ({
    default: () => _jsx("div", { "data-testid": "eye-icon", children: "Eye" }),
}));
vi.mock("../../icons/eye-off", () => ({
    default: () => _jsx("div", { "data-testid": "eye-off-icon", children: "EyeOff" }),
}));
vi.mock("../../icons/enter-key", () => ({
    default: () => _jsx("div", { "data-testid": "enter-key-icon", children: "Enter" }),
}));
vi.mock("../../icons/loading", () => ({
    default: () => _jsx("div", { "data-testid": "loading-icon", children: "Loading" }),
}));
describe("ResetConfirm", () => {
    const mockContext = {
        lang: "fr",
        edenApi: "https://api.test.com",
        site: "test-site",
        config: {
            langPrefix: false,
            edenApi: "",
            site: "",
            lang: "fr",
            langs: ["fr"],
            domains: {},
            siteName: "",
            preProd: false,
            layout: "",
            layouts: [],
        },
        pageProps: {},
        zones: {},
        isConnected: false,
        path: "",
        edenMode: null,
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    const mockOnBack = vi.fn();
    beforeEach(() => {
        vi.clearAllMocks();
        vi.useRealTimers();
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
        delete window.location;
        window.location = { href: "" };
        sessionStorage.clear();
        sessionStorage.setItem(CMS.EDEN_TOKEN, "existing-token");
    });
    it("should render reset confirm form with French translations", () => {
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        expect(container.querySelector('input[name="old-password"]')).toBeInTheDocument();
        expect(container.querySelector('input[name="new-password"]')).toBeInTheDocument();
        expect(container.querySelector('input[name="confirm-new-password"]')).toBeInTheDocument();
        expect(screen.getByRole("button", { name: /Confirmer la réinitialisation/i })).toBeInTheDocument();
    });
    it("should render reset confirm form with English translations", () => {
        const { container } = render(_jsx(ResetConfirm, { lang: "en", onBack: mockOnBack }));
        expect(container.querySelector('input[name="old-password"]')).toBeInTheDocument();
        expect(container.querySelector('input[name="new-password"]')).toBeInTheDocument();
        expect(container.querySelector('input[name="confirm-new-password"]')).toBeInTheDocument();
        expect(screen.getByRole("button", { name: /Confirm reset/i })).toBeInTheDocument();
    });
    it("should render logo", () => {
        render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        expect(screen.getByTestId("logo")).toBeInTheDocument();
    });
    it("should update old password input", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const oldPasswordInput = container.querySelector('input[name="old-password"]');
        await user.type(oldPasswordInput, "oldpass123");
        expect(oldPasswordInput.value).toBe("oldpass123");
    });
    it("should update new password input", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const newPasswordInput = container.querySelector('input[name="new-password"]');
        await user.type(newPasswordInput, "newpass123");
        expect(newPasswordInput.value).toBe("newpass123");
    });
    it("should update confirm password input", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const confirmPasswordInput = container.querySelector('input[name="confirm-new-password"]');
        await user.type(confirmPasswordInput, "newpass123");
        expect(confirmPasswordInput.value).toBe("newpass123");
    });
    it("should have autofocus on old password input", () => {
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const oldPasswordInput = container.querySelector('input[name="old-password"]');
        expect(oldPasswordInput).toHaveFocus();
    });
    it("should have correct autocomplete attributes", () => {
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const oldPasswordInput = container.querySelector('input[name="old-password"]');
        const newPasswordInput = container.querySelector('input[name="new-password"]');
        const confirmPasswordInput = container.querySelector('input[name="confirm-new-password"]');
        expect(oldPasswordInput).toHaveAttribute("autocomplete", "current-password");
        expect(newPasswordInput).toHaveAttribute("autocomplete", "new-password");
        expect(confirmPasswordInput).toHaveAttribute("autocomplete", "new-password");
    });
    it("should toggle old password visibility", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const oldPasswordInput = container.querySelector('input[name="old-password"]');
        expect(oldPasswordInput).toHaveAttribute("type", "password");
        const toggleButtons = container.querySelectorAll('button[type="button"]');
        await user.click(toggleButtons[0]);
        expect(oldPasswordInput).toHaveAttribute("type", "text");
    });
    it("should toggle new password visibility", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const newPasswordInput = container.querySelector('input[name="new-password"]');
        expect(newPasswordInput).toHaveAttribute("type", "password");
        const toggleButtons = container.querySelectorAll('button[type="button"]');
        await user.click(toggleButtons[1]);
        expect(newPasswordInput).toHaveAttribute("type", "text");
    });
    it("should toggle confirm password visibility", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const confirmPasswordInput = container.querySelector('input[name="confirm-new-password"]');
        expect(confirmPasswordInput).toHaveAttribute("type", "password");
        const toggleButtons = container.querySelectorAll('button[type="button"]');
        await user.click(toggleButtons[2]);
        expect(confirmPasswordInput).toHaveAttribute("type", "text");
    });
    it("should display error when passwords do not match", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "differentpass");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText(/Les mots de passe ne correspondent pas/i)).toBeInTheDocument();
        });
        expect(API.confirmPasswordReset).not.toHaveBeenCalled();
    });
    it("should display error when session token is missing", async () => {
        const user = userEvent.setup();
        sessionStorage.clear();
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText(/Session expired. Please log in again./i)).toBeInTheDocument();
        });
        expect(API.confirmPasswordReset).not.toHaveBeenCalled();
    });
    it("should submit password reset successfully", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockResolvedValue({
            token: "new-token",
            error: null,
        });
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(API.confirmPasswordReset).toHaveBeenCalledWith("https://api.test.com", "test-site", "oldpass", "newpass123", "existing-token");
        });
        expect(sessionStorage.getItem(CMS.EDEN_TOKEN)).toBe("new-token");
        expect(screen.getByText(/Mot de passe réinitialisé avec succès/i)).toBeInTheDocument();
    });
    it("should clear form fields after successful submission", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockResolvedValue({
            token: "new-token",
            error: null,
        });
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const oldPasswordInput = container.querySelector('input[name="old-password"]');
        const newPasswordInput = container.querySelector('input[name="new-password"]');
        const confirmPasswordInput = container.querySelector('input[name="confirm-new-password"]');
        await user.type(oldPasswordInput, "oldpass");
        await user.type(newPasswordInput, "newpass123");
        await user.type(confirmPasswordInput, "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(oldPasswordInput.value).toBe("");
            expect(newPasswordInput.value).toBe("");
            expect(confirmPasswordInput.value).toBe("");
        });
    });
    it("should redirect after successful password reset", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockResolvedValue({
            token: "new-token",
            error: null,
        });
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText(/Mot de passe réinitialisé avec succès/i)).toBeInTheDocument();
        });
        await waitFor(() => {
            expect(window.location.href).toBe("/");
        }, { timeout: 2000 });
    });
    it("should redirect with langPrefix after successful password reset", async () => {
        const user = userEvent.setup();
        const contextWithLangPrefix = {
            ...mockContext,
            config: { ...mockContext.config, langPrefix: true },
        };
        vi.mocked(Context.useEdenContext).mockReturnValue(contextWithLangPrefix);
        vi.mocked(API.confirmPasswordReset).mockResolvedValue({
            token: "new-token",
            error: null,
        });
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText(/Mot de passe réinitialisé avec succès/i)).toBeInTheDocument();
        });
        await waitFor(() => {
            expect(window.location.href).toBe("/fr/");
        }, { timeout: 2000 });
    });
    it("should display error on invalid credentials", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockResolvedValue({
            token: "",
            error: { message: "Invalid credentials" },
        });
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "wrongpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText(/E-mail ou mot de passe incorrect/i)).toBeInTheDocument();
        });
    });
    it("should display generic error message", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockResolvedValue({
            token: "",
            error: { message: "Network error" },
        });
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText("Network error")).toBeInTheDocument();
        });
    });
    it("should handle exception during password reset", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockRejectedValue(new Error("Network failure"));
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText("An error occurred. Please try again.")).toBeInTheDocument();
        });
    });
    it("should show loading state during submission", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockImplementation(() => new Promise((resolve) => setTimeout(() => resolve({ token: "new-token", error: null }), 100)));
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        expect(submitButton).toBeDisabled();
        expect(screen.getByTestId("loading-icon")).toBeInTheDocument();
    });
    it("should call onBack when back button is clicked", async () => {
        const user = userEvent.setup();
        render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const backButton = screen.getByRole("button", { name: /Se connecter/i });
        await user.click(backButton);
        expect(mockOnBack).toHaveBeenCalledTimes(1);
    });
    it("should not render back button when onBack is not provided", () => {
        render(_jsx(ResetConfirm, { lang: "fr" }));
        expect(screen.queryByRole("button", { name: /Se connecter/i })).not.toBeInTheDocument();
    });
    it("should not render back button after success", async () => {
        const user = userEvent.setup();
        vi.mocked(API.confirmPasswordReset).mockResolvedValue({
            token: "new-token",
            error: null,
        });
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        await user.type(container.querySelector('input[name="old-password"]'), "oldpass");
        await user.type(container.querySelector('input[name="new-password"]'), "newpass123");
        await user.type(container.querySelector('input[name="confirm-new-password"]'), "newpass123");
        const submitButton = screen.getByRole("button", {
            name: /Confirmer la réinitialisation/i,
        });
        await user.click(submitButton);
        await waitFor(() => {
            expect(screen.getByText(/Mot de passe réinitialisé avec succès/i)).toBeInTheDocument();
        });
        expect(screen.queryByRole("button", { name: /Se connecter/i })).not.toBeInTheDocument();
    });
    it("should require all password inputs", () => {
        const { container } = render(_jsx(ResetConfirm, { lang: "fr", onBack: mockOnBack }));
        const oldPasswordInput = container.querySelector('input[name="old-password"]');
        const newPasswordInput = container.querySelector('input[name="new-password"]');
        const confirmPasswordInput = container.querySelector('input[name="confirm-new-password"]');
        expect(oldPasswordInput).toHaveAttribute("required");
        expect(newPasswordInput).toHaveAttribute("required");
        expect(confirmPasswordInput).toHaveAttribute("required");
    });
});
