import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, screen, waitFor } from "@testing-library/react";
import userEvent from "@testing-library/user-event";
import Login from "../login";
import * as Context from "../../context";
import * as CMS from "../../../services/eden";
import * as API from "../../../services/api";
vi.mock("../../context", () => ({
    useEdenContext: vi.fn(),
}));
vi.mock("../../../services/eden", () => ({
    EDEN_TOKEN: "eden-token",
}));
vi.mock("../../../services/api", () => ({
    login: vi.fn(),
    updateDraftMode: vi.fn(),
}));
vi.mock("../../logo", () => ({
    default: () => _jsx("div", { "data-testid": "logo", children: "Logo" }),
}));
vi.mock("../../icons/eye", () => ({
    default: () => _jsx("div", { "data-testid": "eye-icon", children: "Eye" }),
}));
vi.mock("../../icons/eye-off", () => ({
    default: () => _jsx("div", { "data-testid": "eye-off-icon", children: "EyeOff" }),
}));
vi.mock("../../icons/enter-key", () => ({
    default: () => _jsx("div", { "data-testid": "enter-key-icon", children: "Enter" }),
}));
vi.mock("../../icons/loading", () => ({
    default: () => _jsx("div", { "data-testid": "loading-icon", children: "Loading" }),
}));
vi.mock("../reset", () => ({
    default: () => _jsx("div", { "data-testid": "reset-form", children: "Reset Form" }),
}));
vi.mock("../reset-confirm", () => ({
    default: () => _jsx("div", { "data-testid": "reset-confirm-form", children: "Reset Confirm Form" }),
}));
describe("Login", () => {
    const mockContext = {
        lang: "fr",
        edenApi: "https://api.test.com",
        site: "test-site",
        config: {
            langPrefix: false,
            edenApi: "",
            site: "",
            lang: "fr",
            langs: ["fr"],
            domains: {},
            siteName: "",
            preProd: false,
            layout: "",
            layouts: [],
        },
        pageProps: {},
        zones: {},
        isConnected: false,
        path: "",
        edenMode: null,
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
        delete window.location;
        window.location = { href: "" };
        Object.defineProperty(navigator, "language", {
            writable: true,
            configurable: true,
            value: "fr-FR",
        });
    });
    it("should render login form", () => {
        const { container } = render(_jsx(Login, {}));
        expect(screen.getByLabelText(/E-mail/i)).toBeInTheDocument();
        expect(container.querySelector('input[name="password"]')).toBeInTheDocument();
        expect(screen.getByRole("button", { name: /Se connecter/i })).toBeInTheDocument();
    });
    it("should render logo", () => {
        render(_jsx(Login, {}));
        expect(screen.getByTestId("logo")).toBeInTheDocument();
    });
    it("should toggle password visibility", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(Login, {}));
        const passwordInput = container.querySelector('input[name="password"]');
        expect(passwordInput).toHaveAttribute("type", "password");
        expect(screen.getByTestId("eye-off-icon")).toBeInTheDocument();
        const toggleButton = screen.getAllByRole("button")[0];
        await user.click(toggleButton);
        expect(passwordInput).toHaveAttribute("type", "text");
        expect(screen.getByTestId("eye-icon")).toBeInTheDocument();
    });
    it("should update email input", async () => {
        const user = userEvent.setup();
        render(_jsx(Login, {}));
        const emailInput = screen.getByLabelText(/E-mail/i);
        await user.type(emailInput, "test@example.com");
        expect(emailInput.value).toBe("test@example.com");
    });
    it("should update password input", async () => {
        const user = userEvent.setup();
        const { container } = render(_jsx(Login, {}));
        const passwordInput = container.querySelector('input[name="password"]');
        await user.type(passwordInput, "password123");
        expect(passwordInput.value).toBe("password123");
    });
    it("should submit login form successfully", async () => {
        const user = userEvent.setup();
        vi.mocked(API.login).mockResolvedValue({
            token: "test-token",
            requiresPasswordChange: false,
            error: null,
        });
        vi.mocked(API.updateDraftMode).mockResolvedValue({});
        const { container } = render(_jsx(Login, {}));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.type(container.querySelector('input[name="password"]'), "password123");
        const submitButton = screen.getByRole("button", { name: /Se connecter/i });
        await user.click(submitButton);
        await waitFor(() => {
            expect(API.login).toHaveBeenCalledWith("https://api.test.com", "test-site", "test@example.com", "password123");
        });
        expect(API.updateDraftMode).toHaveBeenCalledWith(true, "test-token");
        expect(sessionStorage.getItem(CMS.EDEN_TOKEN)).toBe("test-token");
    });
    it("should display error on invalid credentials", async () => {
        const user = userEvent.setup();
        vi.mocked(API.login).mockResolvedValue({
            token: "",
            requiresPasswordChange: false,
            error: { message: "Invalid credentials" },
        });
        const { container } = render(_jsx(Login, {}));
        await user.type(screen.getByLabelText(/E-mail/i), "wrong@example.com");
        await user.type(container.querySelector('input[name="password"]'), "wrongpass");
        await user.click(screen.getByRole("button", { name: /Se connecter/i }));
        await waitFor(() => {
            expect(screen.getByText(/e-mail ou mot de passe incorrect/i)).toBeInTheDocument();
        });
    });
    it("should display generic error message", async () => {
        const user = userEvent.setup();
        vi.mocked(API.login).mockResolvedValue({
            token: "",
            requiresPasswordChange: false,
            error: { message: "Network error" },
        });
        const { container } = render(_jsx(Login, {}));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.type(container.querySelector('input[name="password"]'), "password");
        await user.click(screen.getByRole("button", { name: /Se connecter/i }));
        await waitFor(() => {
            expect(screen.getByText("Network error")).toBeInTheDocument();
        });
    });
    it("should show loading state during submission", async () => {
        const user = userEvent.setup();
        vi.mocked(API.login).mockImplementation(() => new Promise((resolve) => setTimeout(() => resolve({
            token: "test-token",
            requiresPasswordChange: false,
            error: null,
        }), 100)));
        const { container } = render(_jsx(Login, {}));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.type(container.querySelector('input[name="password"]'), "password");
        const submitButton = screen.getByRole("button", { name: /Se connecter/i });
        await user.click(submitButton);
        expect(submitButton).toBeDisabled();
        expect(screen.getByTestId("loading-icon")).toBeInTheDocument();
    });
    it("should redirect to home with langPrefix", async () => {
        const user = userEvent.setup();
        const contextWithLangPrefix = {
            ...mockContext,
            config: { ...mockContext.config, langPrefix: true },
        };
        vi.mocked(Context.useEdenContext).mockReturnValue(contextWithLangPrefix);
        vi.mocked(API.login).mockResolvedValue({
            token: "test-token",
            requiresPasswordChange: false,
            error: null,
        });
        vi.mocked(API.updateDraftMode).mockResolvedValue({});
        const { container } = render(_jsx(Login, {}));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.type(container.querySelector('input[name="password"]'), "password");
        await user.click(screen.getByRole("button", { name: /Se connecter/i }));
        await waitFor(() => {
            expect(API.login).toHaveBeenCalled();
            expect(API.updateDraftMode).toHaveBeenCalled();
            expect(window.location.href).toBe("/fr/");
        });
    });
    it("should redirect to home without langPrefix", async () => {
        const user = userEvent.setup();
        vi.mocked(API.login).mockResolvedValue({
            token: "test-token",
            requiresPasswordChange: false,
            error: null,
        });
        vi.mocked(API.updateDraftMode).mockResolvedValue({});
        const { container } = render(_jsx(Login, {}));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.type(container.querySelector('input[name="password"]'), "password");
        await user.click(screen.getByRole("button", { name: /Se connecter/i }));
        await waitFor(() => {
            expect(window.location.href).toBe("/");
        });
    });
    it("should have autofocus on email input", () => {
        render(_jsx(Login, {}));
        const emailInput = screen.getByLabelText(/E-mail/i);
        expect(emailInput).toHaveFocus();
    });
    it("should have correct autocomplete attributes", () => {
        const { container } = render(_jsx(Login, {}));
        const emailInput = screen.getByLabelText(/E-mail/i);
        const passwordInput = container.querySelector('input[name="password"]');
        expect(emailInput).toHaveAttribute("autocomplete", "email");
        expect(passwordInput).toHaveAttribute("autocomplete", "current-password");
    });
    it("should show reset-confirm form when requiresPasswordChange is true", async () => {
        const user = userEvent.setup();
        vi.mocked(API.login).mockResolvedValue({
            token: "test-token",
            requiresPasswordChange: true,
            error: null,
        });
        const { container } = render(_jsx(Login, {}));
        await user.type(screen.getByLabelText(/E-mail/i), "test@example.com");
        await user.type(container.querySelector('input[name="password"]'), "password");
        await user.click(screen.getByRole("button", { name: /Se connecter/i }));
        await waitFor(() => {
            expect(screen.getByTestId("reset-confirm-form")).toBeInTheDocument();
        });
        expect(sessionStorage.getItem(CMS.EDEN_TOKEN)).toBe("test-token");
        expect(API.updateDraftMode).not.toHaveBeenCalled();
    });
    it("should show reset form when forgot password button is clicked", async () => {
        const user = userEvent.setup();
        render(_jsx(Login, {}));
        const forgotPasswordButton = screen.getByRole("button", {
            name: /Mot de passe oublié/i,
        });
        await user.click(forgotPasswordButton);
        await waitFor(() => {
            expect(screen.getByTestId("reset-form")).toBeInTheDocument();
        });
    });
    it("should detect English language from navigator", () => {
        Object.defineProperty(navigator, "language", {
            writable: true,
            configurable: true,
            value: "en-US",
        });
        render(_jsx(Login, {}));
        expect(screen.getByLabelText(/Email/i)).toBeInTheDocument();
        expect(screen.getByRole("button", { name: /Log in/i })).toBeInTheDocument();
    });
});
