import { jsx as _jsx, Fragment as _Fragment, jsxs as _jsxs } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render } from "@testing-library/react";
import Zone from "../zone";
import * as Context from "../context";
const mockContext = {
    lang: "en",
    path: "/test",
    zones: {
        header: "<h1>Header Content</h1>",
        footer: "<p>Footer Content</p>",
    },
    config: {},
    pageProps: {},
    isConnected: false,
    edenApi: "",
    site: "",
    edenMode: null,
    menuPath: {},
    breadcrumb: {},
    seo: {},
    metadata: {},
    langSwitches: {},
};
vi.mock("../context", async () => {
    const actual = await vi.importActual("../context");
    return {
        ...actual,
        useEdenContext: vi.fn(),
    };
});
describe("Zone", () => {
    beforeEach(() => {
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
    });
    it("should render zone with default div tag", () => {
        const { container } = render(_jsx(Zone, { code: "header" }));
        expect(container.querySelector("div[data-eden-zone='header']")).toBeInTheDocument();
    });
    it("should render zone with custom tag", () => {
        const { container } = render(_jsx(Zone, { tag: "section", code: "header" }));
        expect(container.querySelector("section[data-eden-zone='header']")).toBeInTheDocument();
    });
    it("should render zone content from zones object", () => {
        const { container } = render(_jsx(Zone, { code: "header" }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone?.innerHTML).toBe("<h1>Header Content</h1>");
    });
    it("should render zone with custom content prop", () => {
        const { container } = render(_jsx(Zone, { code: "header", content: "<div>Custom Content</div>" }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone?.innerHTML).toBe("<div>Custom Content</div>");
    });
    it("should prioritize content prop over zones object", () => {
        const { container } = render(_jsx(Zone, { code: "header", content: "<div>Override Content</div>" }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone?.innerHTML).toBe("<div>Override Content</div>");
    });
    it("should apply className", () => {
        const { container } = render(_jsx(Zone, { code: "header", className: "custom-class" }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveClass("custom-class");
    });
    it("should apply data attributes", () => {
        const { container } = render(_jsx(Zone, { code: "header", tooltip: "Test Tooltip", fixed: true, templates: "template1,template2" }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveAttribute("data-eden-tooltip", "Test Tooltip");
        expect(zone).toHaveAttribute("data-eden-fixed", "true");
        expect(zone).toHaveAttribute("data-eden-templates", "template1,template2");
    });
    it("should apply maxBlocks attribute", () => {
        const { container } = render(_jsx(Zone, { code: "header", maxBlocks: 5 }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveAttribute("data-eden-max-blocks", "5");
    });
    it("should apply maxBlocks as string", () => {
        const { container } = render(_jsx(Zone, { code: "header", maxBlocks: "10" }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveAttribute("data-eden-max-blocks", "10");
    });
    it("should apply tagsLabel and allowedTags attributes", () => {
        const { container } = render(_jsx(Zone, { code: "header", tagsLabel: "Tags", allowedTags: ["tag1", "tag2"], allowMultipleTags: true }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveAttribute("data-eden-tags-label", "Tags");
        expect(zone).toHaveAttribute("data-eden-allowed-tags", "tag1,tag2");
        expect(zone).toHaveAttribute("data-eden-allow-multiple-tags", "true");
    });
    it("should apply defaultContent attribute", () => {
        const { container } = render(_jsx(Zone, { code: "header", defaultContent: "<p>Default</p>" }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveAttribute("data-eden-default-content", "<p>Default</p>");
    });
    it("should apply readOnly attribute and aria-hidden", () => {
        const { container } = render(_jsx(Zone, { code: "header", readOnly: true }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveAttribute("data-eden-read-only", "true");
        expect(zone).toHaveAttribute("aria-hidden", "true");
    });
    it("should apply uuid attribute", () => {
        const { container } = render(_jsx(Zone, { code: "header", uuid: true }));
        const zone = container.querySelector("[data-eden-zone='header']");
        expect(zone).toHaveAttribute("data-eden-uuid", "true");
    });
    it("should render empty string when zone code not found", () => {
        const { container } = render(_jsx(Zone, { code: "non-existent" }));
        const zone = container.querySelector("[data-eden-zone='non-existent']");
        expect(zone?.innerHTML).toBe("");
    });
    it("should use key based on lang and path", () => {
        const { container, rerender } = render(_jsx(Zone, { code: "header" }));
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            lang: "fr",
            path: "/autre",
        });
        rerender(_jsx(Zone, { code: "header" }));
        expect(container.querySelector("[data-eden-zone='header']")).toBeInTheDocument();
    });
    it("should handle multiple zones", () => {
        const { container } = render(_jsxs(_Fragment, { children: [_jsx(Zone, { code: "header" }), _jsx(Zone, { code: "footer" })] }));
        expect(container.querySelector("[data-eden-zone='header']")).toBeInTheDocument();
        expect(container.querySelector("[data-eden-zone='footer']")).toBeInTheDocument();
    });
});
