import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, screen } from "@testing-library/react";
import EdenPage from "../page";
import * as Context from "../context";
import * as UseDateHiddenBlocks from "../../hooks/use-date-hidden-blocks";
import * as UsePatchedInternalLinks from "../../hooks/use-patched-internal-links";
import "@testing-library/jest-dom";
vi.mock("../../hooks/use-date-hidden-blocks", () => ({
    default: vi.fn(),
}));
vi.mock("../../hooks/use-patched-internal-links", () => ({
    default: vi.fn(),
}));
vi.mock("../context", async () => {
    const actual = await vi.importActual("../context");
    return {
        ...actual,
        useEdenContext: vi.fn(),
    };
});
describe("EdenPage", () => {
    const DefaultLayout = (props) => (_jsx("div", { "data-testid": "default-layout", children: "Default Layout" }));
    const CustomLayout = (props) => (_jsx("div", { "data-testid": "custom-layout", children: "Custom Layout" }));
    const mockLayouts = {
        default: DefaultLayout,
        custom: CustomLayout,
    };
    const mockContext = {
        config: {
            layout: "default",
            edenApi: "test-api",
            site: "test-site",
            lang: "en",
            langs: ["en", "fr"],
            domains: {},
            siteName: "Test",
            preProd: false,
            layouts: ["default"],
            langPrefix: false,
        },
        pageProps: {
            title: "Test Page",
            content: "Test Content",
        },
        lang: "en",
        path: "/test",
        zones: {},
        isConnected: false,
        edenApi: "",
        site: "",
        edenMode: null,
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
    });
    it("should render the correct layout based on config", () => {
        render(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(screen.getByTestId("default-layout")).toBeInTheDocument();
    });
    it("should pass pageProps to layout", () => {
        render(_jsx(EdenPage, { layouts: mockLayouts }));
        const layoutElement = screen.getByTestId("default-layout");
        expect(layoutElement.textContent).toBe("Default Layout");
    });
    it("should render custom layout when specified", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            config: { ...mockContext.config, layout: "custom" },
        });
        render(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(screen.getByTestId("custom-layout")).toBeInTheDocument();
    });
    it("should return null when layout not found", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            config: { ...mockContext.config, layout: "non-existent" },
        });
        const { container } = render(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(container.firstChild).toBeNull();
    });
    it("should call useDateHiddenBlocks hook", () => {
        render(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(UseDateHiddenBlocks.default).toHaveBeenCalled();
    });
    it("should call usePatchedInternalLinks hook", () => {
        render(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(UsePatchedInternalLinks.default).toHaveBeenCalled();
    });
    it("should handle empty pageProps", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            pageProps: {},
        });
        render(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(screen.getByTestId("default-layout")).toBeInTheDocument();
    });
    it("should re-render when layout changes", () => {
        const { rerender } = render(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(screen.getByTestId("default-layout")).toBeInTheDocument();
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            config: { ...mockContext.config, layout: "custom" },
        });
        rerender(_jsx(EdenPage, { layouts: mockLayouts }));
        expect(screen.getByTestId("custom-layout")).toBeInTheDocument();
    });
});
