import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, waitFor } from "@testing-library/react";
import EdenLoader from "../loader";
import * as Context from "../context";
import * as Eden from "../../services/eden";
import * as UsePatchedInternalLinks from "../../hooks/use-patched-internal-links";
import * as UseDateHiddenBlocks from "../../hooks/use-date-hidden-blocks";
vi.mock("../context", async () => {
    const actual = await vi.importActual("../context");
    return {
        ...actual,
        useEdenContext: vi.fn(),
    };
});
vi.mock("../../hooks/use-patched-internal-links", () => ({
    default: vi.fn(),
}));
vi.mock("../../hooks/use-date-hidden-blocks", () => ({
    default: vi.fn(),
}));
vi.mock("../../services/eden", () => ({
    setup: vi.fn(),
}));
vi.mock("../../@editing-tools/eden", () => ({
    default: vi.fn().mockImplementation(() => ({
        start: vi.fn().mockResolvedValue(undefined),
    })),
}));
describe("EdenLoader", () => {
    const mockConfig = {
        edenApi: "https://api.test.com",
        site: "test-site",
        lang: "en",
        langs: ["en", "fr"],
        domains: {},
        siteName: "Test Site",
        preProd: false,
        layout: "default",
        layouts: [],
        langPrefix: false,
    };
    const mockContext = {
        isConnected: false,
        lang: "en",
        config: mockConfig,
        pageProps: {},
        zones: {},
        edenApi: "",
        site: "",
        path: "",
        edenMode: null,
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
        delete window.Eden;
    });
    it("should render null", () => {
        const { container } = render(_jsx(EdenLoader, { config: mockConfig }));
        expect(container.firstChild).toBeNull();
    });
    it("should call usePatchedInternalLinks hook", () => {
        render(_jsx(EdenLoader, { config: mockConfig }));
        expect(UsePatchedInternalLinks.default).toHaveBeenCalled();
    });
    it("should call useDateHiddenBlocks hook", () => {
        render(_jsx(EdenLoader, { config: mockConfig }));
        expect(UseDateHiddenBlocks.default).toHaveBeenCalled();
    });
    it("should not load Eden when not connected", () => {
        render(_jsx(EdenLoader, { config: mockConfig }));
        expect(Eden.setup).not.toHaveBeenCalled();
    });
    it("should load Eden when connected", async () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            isConnected: true,
        });
        render(_jsx(EdenLoader, { config: mockConfig }));
        await waitFor(() => {
            expect(Eden.setup).toHaveBeenCalledWith(mockConfig);
        });
    });
    it("should initialize Eden CMS when connected", async () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            isConnected: true,
        });
        const mockStart = vi.fn().mockResolvedValue(undefined);
        const MockEdenCms = vi.fn().mockImplementation(() => ({
            start: mockStart,
        }));
        vi.doMock("../../@editing-tools/eden", () => ({
            default: MockEdenCms,
        }));
        render(_jsx(EdenLoader, { config: mockConfig }));
        await waitFor(() => {
            expect(Eden.setup).toHaveBeenCalled();
        });
    });
    it("should set window.Eden when connected", async () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            isConnected: true,
        });
        render(_jsx(EdenLoader, { config: mockConfig }));
        await waitFor(() => {
            expect(Eden.setup).toHaveBeenCalled();
        }, { timeout: 1000 });
    });
    it("should handle connection state change", async () => {
        const { rerender } = render(_jsx(EdenLoader, { config: mockConfig }));
        expect(Eden.setup).not.toHaveBeenCalled();
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            isConnected: true,
        });
        rerender(_jsx(EdenLoader, { config: mockConfig }));
        await waitFor(() => {
            expect(Eden.setup).toHaveBeenCalledWith(mockConfig);
        });
    });
    it("should pass config to Eden setup", async () => {
        const customConfig = {
            ...mockConfig,
            site: "custom-site",
            edenApi: "https://custom-api.com",
        };
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            isConnected: true,
        });
        render(_jsx(EdenLoader, { config: customConfig }));
        await waitFor(() => {
            expect(Eden.setup).toHaveBeenCalledWith(customConfig);
        });
    });
});
