import { jsx as _jsx } from "react/jsx-runtime";
import { describe, it, expect, vi, beforeEach } from "vitest";
import { render, screen } from "@testing-library/react";
import Link from "../link";
import * as Context from "../context";
import * as Path from "../../utils/path";
vi.mock("next/link", () => ({
    default: ({ children, href, ...props }) => (_jsx("a", { href: typeof href === "string" ? href : href.pathname, ...props, children: children })),
    LinkProps: {},
}));
vi.mock("../context", async () => {
    const actual = await vi.importActual("../context");
    return {
        ...actual,
        useEdenContext: vi.fn(),
    };
});
vi.mock("../../utils/path", async () => {
    const actual = await vi.importActual("../../utils/path");
    return {
        ...actual,
        toStatic: vi.fn((path) => path),
        toLocalized: vi.fn((path) => path),
    };
});
describe("Link", () => {
    const mockContext = {
        lang: "en",
        config: {
            langPrefix: false,
            edenApi: "",
            site: "",
            lang: "en",
            langs: ["en"],
            domains: {},
            siteName: "",
            preProd: false,
            layout: "",
            layouts: [],
        },
        isConnected: false,
        pageProps: {},
        zones: {},
        edenApi: "",
        site: "",
        path: "",
        edenMode: null,
        menuPath: {},
        breadcrumb: {},
        seo: {},
        metadata: {},
        langSwitches: {},
    };
    beforeEach(() => {
        vi.clearAllMocks();
        vi.mocked(Context.useEdenContext).mockReturnValue(mockContext);
        vi.mocked(Path.toStatic).mockImplementation((path) => path);
        vi.mocked(Path.toLocalized).mockImplementation((path) => path);
    });
    it("should render link with href", () => {
        render(_jsx(Link, { href: "/test", children: "Test Link" }));
        const link = screen.getByText("Test Link");
        expect(link).toBeInTheDocument();
        expect(link.tagName).toBe("A");
    });
    it("should render link with title prop", () => {
        render(_jsx(Link, { href: "/test", title: "<strong>Bold Title</strong>" }));
        const link = screen.getByRole("link");
        expect(link.innerHTML).toBe("<strong>Bold Title</strong>");
    });
    it("should prioritize children over title prop", () => {
        render(_jsx(Link, { href: "/test", title: "Title", children: "Children Content" }));
        expect(screen.getByText("Children Content")).toBeInTheDocument();
    });
    it("should handle external links with http", () => {
        vi.mocked(Path.toStatic).mockReturnValue("http://external.com");
        render(_jsx(Link, { href: "http://external.com", children: "External" }));
        const link = screen.getByText("External");
        expect(link).toHaveAttribute("target", "_blank");
        expect(link).toHaveAttribute("rel", "noopener noreferrer");
    });
    it("should handle external links with https", () => {
        vi.mocked(Path.toStatic).mockReturnValue("https://external.com");
        render(_jsx(Link, { href: "https://external.com", children: "Secure External" }));
        const link = screen.getByText("Secure External");
        expect(link).toHaveAttribute("target", "_blank");
        expect(link).toHaveAttribute("rel", "noopener noreferrer");
    });
    it("should not add target/rel for internal links", () => {
        vi.mocked(Path.toStatic).mockReturnValue("/internal");
        render(_jsx(Link, { href: "/internal", children: "Internal" }));
        const link = screen.getByText("Internal");
        expect(link).not.toHaveAttribute("target");
        expect(link).not.toHaveAttribute("rel");
    });
    it("should render regular anchor when connected", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            isConnected: true,
        });
        render(_jsx(Link, { href: "/test", children: "Connected Link" }));
        const link = screen.getByText("Connected Link");
        expect(link.tagName).toBe("A");
    });
    it("should handle object href with pathname", () => {
        const hrefObject = { pathname: "/test", query: { id: "1" } };
        render(_jsx(Link, { href: hrefObject, children: "Object Link" }));
        expect(screen.getByText("Object Link")).toBeInTheDocument();
    });
    it("should localize internal paths", () => {
        vi.mocked(Path.toStatic).mockReturnValue("/test");
        vi.mocked(Path.toLocalized).mockReturnValue("/en/test");
        render(_jsx(Link, { href: "/test", children: "Localized" }));
        expect(Path.toLocalized).toHaveBeenCalledWith("/test", "en", false);
    });
    it("should handle langPrefix in config", () => {
        vi.mocked(Context.useEdenContext).mockReturnValue({
            ...mockContext,
            config: { ...mockContext.config, langPrefix: true },
        });
        vi.mocked(Path.toStatic).mockReturnValue("/test");
        vi.mocked(Path.toLocalized).mockReturnValue("/en/test");
        render(_jsx(Link, { href: "/test", children: "Prefixed" }));
        expect(Path.toLocalized).toHaveBeenCalledWith("/test", "en", true);
    });
    it("should pass through additional props", () => {
        render(_jsx(Link, { href: "/test", className: "custom-class", "data-testid": "custom-link", children: "Custom Props" }));
        const link = screen.getByTestId("custom-link");
        expect(link).toHaveClass("custom-class");
    });
    it("should handle as prop", () => {
        render(_jsx(Link, { href: "/test", as: "/custom", children: "As Prop" }));
        expect(screen.getByText("As Prop")).toBeInTheDocument();
    });
});
