import EdenModal from "../common/eden-modal.js";
import ImageSettingsModal from "./image-settings-modal.js";
import FileManagerModal from "./file-manager-modal.js";
import html from "../../services/html.js";
import t from "../../i18n/index.js";
export default class SlideshowModal {
    constructor(onChange, images, width, height, credits) {
        this.onUpdate = () => {
            const root = this.modal.shadowRoot;
            const nodes = Array.from(root.querySelectorAll(".slideshow-builder__image"));
            const images = nodes
                .filter((_, index) => index !== nodes.length - 1)
                .map((node) => ({
                src: node.src,
                alt: node.alt || "",
                metadata: node.getAttribute("data-metadata"),
            }));
            const creditsInput = root.querySelector('[name="credits"]');
            const credits = creditsInput?.value || "";
            this.onChange(images, credits);
        };
        this.removeImg = (e) => {
            const target = e.target;
            const item = target.closest(".slideshow-builder__item");
            if (item !== null) {
                item.remove();
            }
        };
        this.editAlt = (e) => {
            const target = e.target;
            const item = target.closest(".slideshow-builder__item");
            if (item !== null) {
                const image = item.querySelector('input[type="image"]');
                if (!image)
                    return;
                let description = image.getAttribute("alt") || "";
                if (description === "undefined") {
                    description = "";
                }
                new ImageSettingsModal(this.updateAlt(image), description).init();
            }
        };
        this.updateAlt = (image) => (description) => {
            if (description) {
                image.setAttribute("alt", description);
            }
            else {
                image.removeAttribute("alt");
            }
        };
        this.getPath = (target) => (path, metadata) => {
            if (target === target.parentNode?.lastElementChild) {
                target.insertAdjacentHTML("beforebegin", imgTemplate({ src: path, alt: "", metadata }));
            }
            else {
                const parent = target.closest(".slideshow-builder__item");
                if (!parent)
                    return;
                const image = parent.querySelector('input[type="image"]');
                image.setAttribute("src", path);
                image.setAttribute("data-metadata", metadata);
            }
        };
        this.openCropFilePicker = (e) => {
            const target = e.target;
            const button = target.closest(".slideshow-builder__button--crop");
            if (!button)
                return;
            const parent = button.closest(".slideshow-builder__item");
            if (!parent)
                return;
            const image = parent.querySelector('input[type="image"]');
            if (!image)
                return;
            const metadata = image.getAttribute("data-metadata");
            if (button !== null) {
                new FileManagerModal({
                    callback: this.getPath(button),
                    width: this.imgWidth,
                    height: this.imgHeight,
                    metadata,
                });
            }
        };
        this.openFilePicker = (e) => {
            const target = e.target;
            const button = target.closest(".slideshow-builder__item, .slideshow-builder__image--add-image");
            if (button !== null) {
                new FileManagerModal({
                    callback: this.getPath(button),
                    width: this.imgWidth,
                    height: this.imgHeight,
                });
            }
        };
        this.onChange = onChange;
        this.images = images;
        this.imgWidth = width;
        this.imgHeight = height;
        this.credits = credits;
        this.modal = new EdenModal({
            icon: "image",
            title: t("modals.slideshow.title"),
            content: modalTemplate(this.images, this.credits),
            buttons: [
                { value: t("common.cancel"), cssClass: "cancel" },
                {
                    value: t("common.validate"),
                    onClick: this.onUpdate,
                    cssClass: "confirm",
                    autofocus: true,
                },
            ],
        });
        document.body.append(this.modal);
        this.modal.shadowRoot.addEventListener("click", (e) => {
            const target = e.target;
            if (!target)
                return;
            const targetClassList = target.classList;
            if (targetClassList.contains("slideshow-builder__button--edit") ||
                targetClassList.contains("slideshow-builder__image--add-image")) {
                this.openFilePicker(e);
            }
            if (targetClassList.contains("slideshow-builder__button--crop")) {
                this.openCropFilePicker(e);
            }
            if (targetClassList.contains("slideshow-builder__button--delete")) {
                this.removeImg(e);
            }
            if (targetClassList.contains("slideshow-builder__button--alt")) {
                this.editAlt(e);
            }
        });
    }
}
function modalTemplate(images, credits) {
    return html `
    <style>
      .slideshow-builder {
        --item-size: 160px;
        --item-radius: var(--eden-radius-200, 8px);

        width: 740px;
        max-width: 100%;
        padding: 25px 5px 5px 10px;
        box-sizing: border-box;
      }

      .slideshow-builder__list {
        display: flex;
        flex-wrap: wrap;
        gap: 20px;
        margin: 0 0 16px 0;
        padding: 0;
      }

      .slideshow-builder__item {
        position: relative;
        flex: 0 0 var(--item-size);
        height: var(--item-size);
        padding: 0;
        border: 0;
        border-radius: var(--item-radius);
        cursor: pointer;
        user-select: none;
      }

      .slideshow-builder__item:focus {
        outline: none;
      }

      .slideshow-builder__item:focus-visible {
        outline: 2px solid var(--eden-secondary-color);
        outline-offset: 3px;
      }

      .slideshow-builder__edit-icon {
        width: 30px;
        height: 30px;
        fill: var(--eden-grey-000);
        pointer-events: none;
      }

      .slideshow-builder__image {
        display: block;
        width: 100%;
        height: 100%;
        padding: 0;
        object-fit: cover;
        border: 0;
        border-radius: var(--item-radius);
        overflow: hidden;
      }

      .slideshow-builder__image:focus {
        outline: none;
      }

      .slideshow-builder__image:focus-visible {
        outline: 2px solid var(--eden-secondary-color);
        outline-offset: 3px;
      }

      .slideshow-builder__buttons {
        position: absolute;
        bottom: 8px;
        right: 8px;
        display: flex;
        gap: 6px;
      }

      .slideshow-builder__button {
        width: 26px;
        height: 26px;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        color: var(--eden-grey-000);
        border: none;
        border-radius: var(--eden-radius-100);
      }

      .slideshow-builder__button:focus {
        outline: none;
      }

      .slideshow-builder__button:focus-visible {
        outline: 2px solid var(--eden-secondary-color);
        outline-offset: 3px;
      }

      .slideshow-builder__button--edit,
      .slideshow-builder__button--crop,
      .slideshow-builder__button--alt {
        background-color: var(--eden-primary-color);
      }

      .slideshow-builder__image:not([data-metadata])
        + .slideshow-builder__buttons
        .slideshow-builder__button--crop {
        display: none;
      }

      .slideshow-builder__button--delete {
        background-color: var(--eden-red-400);
      }

      .slideshow-builder__icon {
        width: 20px;
        height: 20px;
        fill: currentcolor;
        pointer-events: none;
      }

      .slideshow-builder__image--add-image {
        width: var(--item-size);
        height: var(--item-size);
        display: flex;
        align-items: center;
        justify-content: center;
        background-color: var(--eden-grey-100);
        border: 2px dashed var(--eden-grey-300);
        border-radius: var(--item-radius);
        cursor: pointer;
        transition: background-color 160ms var(--eden-transition-easing);
      }

      .slideshow-builder__add-icon {
        width: 40px;
        height: 40px;
        fill: var(--eden-grey-500);
        pointer-events: none;
      }
    </style>
    <div class="slideshow-builder">
      <div class="slideshow-builder__list">
        ${images.map(imgTemplate).join("")}
        <button
          class="slideshow-builder__image slideshow-builder__image--add-image"
        >
          <svg viewBox="0 0 24 24" class="slideshow-builder__add-icon">
            <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
          </svg>
        </button>
      </div>
      <eden-input
        type="text"
        label="${t("modals.slideshow.credits")}"
        name="credits"
        value="${credits || ""}"
      ></eden-input>
    </div>
  `;
}
function imgTemplate(image) {
    const src = image.src;
    const alt = image.alt || "";
    const metadata = image.metadata;
    const metadataAttribute = metadata ? `data-metadata="${metadata}"` : "";
    return html `
    <div class="slideshow-builder__item">
      <input
        type="image"
        class="slideshow-builder__image"
        src="${src}"
        alt="${alt}"
        ${metadataAttribute}
      />
      <div class="slideshow-builder__buttons">
        <button
          is="eden-button"
          tooltip="${t("modals.slideshow.replace-image")}"
          class="slideshow-builder__button slideshow-builder__button--edit"
        >
          <eden-icon code="image" class="slideshow-builder__icon"></eden-icon>
        </button>
        <button
          is="eden-button"
          tooltip="${t("modals.slideshow.crop-image")}"
          class="slideshow-builder__button slideshow-builder__button--crop"
        >
          <eden-icon code="crop" class="slideshow-builder__icon"></eden-icon>
        </button>
        <button
          is="eden-button"
          class="slideshow-builder__button slideshow-builder__button--alt"
          tooltip="${t("modals.slideshow.add-edit-description")}"
        >
          <eden-icon
            code="description"
            class="slideshow-builder__icon"
          ></eden-icon>
        </button>
        <button
          is="eden-button"
          class="slideshow-builder__button slideshow-builder__button--delete"
          tooltip="${t("modals.slideshow.remove-image")}"
        >
          <eden-icon code="delete" class="slideshow-builder__icon"></eden-icon>
        </button>
      </div>
    </div>
  `;
}
