import { describe, it, expect, beforeEach, vi } from "vitest";
import { smoothScrollTo, isLocalhost, disableBrowserTranslation, } from "../util";
describe("Util Service", () => {
    describe("smoothScrollTo", () => {
        beforeEach(() => {
            window.scroll = vi.fn();
            window.innerHeight = 800;
        });
        it("should scroll to element with smooth behavior", () => {
            const element = document.createElement("div");
            document.body.appendChild(element);
            Object.defineProperty(element, "offsetTop", {
                value: 1000,
                writable: true,
            });
            Object.defineProperty(element, "offsetLeft", {
                value: 0,
                writable: true,
            });
            Object.defineProperty(element, "scrollTop", { value: 0, writable: true });
            Object.defineProperty(element, "scrollLeft", {
                value: 0,
                writable: true,
            });
            Object.defineProperty(element, "offsetParent", {
                value: null,
                writable: true,
            });
            smoothScrollTo(element);
            expect(window.scroll).toHaveBeenCalledWith({
                top: expect.any(Number),
                behavior: "smooth",
            });
            document.body.removeChild(element);
        });
        it("should calculate position accounting for window height", () => {
            const element = document.createElement("div");
            document.body.appendChild(element);
            Object.defineProperty(element, "offsetTop", {
                value: 1000,
                writable: true,
            });
            Object.defineProperty(element, "offsetLeft", {
                value: 0,
                writable: true,
            });
            Object.defineProperty(element, "scrollTop", { value: 0, writable: true });
            Object.defineProperty(element, "scrollLeft", {
                value: 0,
                writable: true,
            });
            Object.defineProperty(element, "offsetParent", {
                value: null,
                writable: true,
            });
            smoothScrollTo(element);
            expect(window.scroll).toHaveBeenCalledWith({
                top: 800,
                behavior: "smooth",
            });
            document.body.removeChild(element);
        });
    });
    describe("isLocalhost", () => {
        it("should return true when hostname is localhost", () => {
            Object.defineProperty(window, "location", {
                value: { hostname: "localhost" },
                writable: true,
                configurable: true,
            });
            expect(isLocalhost()).toBe(true);
        });
        it("should return false when hostname is not localhost", () => {
            Object.defineProperty(window, "location", {
                value: { hostname: "example.com" },
                writable: true,
                configurable: true,
            });
            expect(isLocalhost()).toBe(false);
        });
        it("should return false for 127.0.0.1", () => {
            Object.defineProperty(window, "location", {
                value: { hostname: "127.0.0.1" },
                writable: true,
                configurable: true,
            });
            expect(isLocalhost()).toBe(false);
        });
        it("should return false for production domains", () => {
            Object.defineProperty(window, "location", {
                value: { hostname: "www.example.com" },
                writable: true,
                configurable: true,
            });
            expect(isLocalhost()).toBe(false);
        });
    });
    describe("disableBrowserTranslation", () => {
        beforeEach(() => {
            document.documentElement.removeAttribute("translate");
            document.documentElement.classList.remove("notranslate");
        });
        it("should set translate attribute to no", () => {
            disableBrowserTranslation();
            expect(document.documentElement.getAttribute("translate")).toBe("no");
        });
        it("should add notranslate class", () => {
            disableBrowserTranslation();
            expect(document.documentElement.classList.contains("notranslate")).toBe(true);
        });
        it("should set both attribute and class", () => {
            disableBrowserTranslation();
            expect(document.documentElement.getAttribute("translate")).toBe("no");
            expect(document.documentElement.classList.contains("notranslate")).toBe(true);
        });
        it("should not remove existing classes", () => {
            document.documentElement.classList.add("existing-class");
            disableBrowserTranslation();
            expect(document.documentElement.classList.contains("existing-class")).toBe(true);
            expect(document.documentElement.classList.contains("notranslate")).toBe(true);
        });
        it("should be idempotent", () => {
            disableBrowserTranslation();
            disableBrowserTranslation();
            disableBrowserTranslation();
            expect(document.documentElement.getAttribute("translate")).toBe("no");
            expect(Array.from(document.documentElement.classList).filter((c) => c === "notranslate").length).toBe(1);
        });
    });
});
