import { describe, it, expect, beforeEach } from "vitest";
import { decodeToken } from "../token";
describe("Token Service", () => {
    beforeEach(() => {
        window.sessionStorage.clear();
    });
    describe("decodeToken", () => {
        it("should decode valid JWT-like token", () => {
            const payload = { userId: 123, role: "admin" };
            const encodedPayload = btoa(unescape(encodeURIComponent(JSON.stringify(payload))));
            const token = `header.${encodedPayload}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toEqual(payload);
        });
        it("should return undefined if no token exists", () => {
            const result = decodeToken();
            expect(result).toBeUndefined();
        });
        it("should return undefined for invalid token format", () => {
            window.sessionStorage.setItem("eden_token", "invalid-token");
            const result = decodeToken();
            expect(result).toBeUndefined();
        });
        it("should return undefined for token without enough parts", () => {
            window.sessionStorage.setItem("eden_token", "header.payload");
            const result = decodeToken();
            expect(result).toBeUndefined();
        });
        it("should return undefined for non-JSON payload", () => {
            const encodedPayload = btoa("not-json-content");
            const token = `header.${encodedPayload}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toBeUndefined();
        });
        it("should handle token with complex JSON payload", () => {
            const payload = {
                userId: 123,
                username: "john_doe",
                roles: ["admin", "editor"],
                metadata: {
                    createdAt: "2024-01-01",
                    permissions: ["read", "write"],
                },
            };
            const encodedPayload = btoa(unescape(encodeURIComponent(JSON.stringify(payload))));
            const token = `header.${encodedPayload}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toEqual(payload);
        });
        it("should handle token with special characters in payload", () => {
            const payload = {
                name: "Jean-François",
                email: "test@example.com",
                message: "Hello, world! 你好",
            };
            const encodedPayload = btoa(unescape(encodeURIComponent(JSON.stringify(payload))));
            const token = `header.${encodedPayload}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toEqual(payload);
        });
        it("should return undefined for empty string token", () => {
            window.sessionStorage.setItem("eden_token", "");
            const result = decodeToken();
            expect(result).toBeUndefined();
        });
        it("should return undefined for malformed base64", () => {
            const validBase64InvalidJson = btoa("not-json-object");
            const token = `header.${validBase64InvalidJson}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toBeUndefined();
        });
        it("should handle numeric values in payload", () => {
            const payload = {
                id: 42,
                score: 98.5,
                count: 0,
            };
            const encodedPayload = btoa(unescape(encodeURIComponent(JSON.stringify(payload))));
            const token = `header.${encodedPayload}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toEqual(payload);
        });
        it("should handle boolean values in payload", () => {
            const payload = {
                active: true,
                verified: false,
                premium: true,
            };
            const encodedPayload = btoa(unescape(encodeURIComponent(JSON.stringify(payload))));
            const token = `header.${encodedPayload}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toEqual(payload);
        });
        it("should handle null values in payload", () => {
            const payload = {
                name: "John",
                middleName: null,
                age: 30,
            };
            const encodedPayload = btoa(unescape(encodeURIComponent(JSON.stringify(payload))));
            const token = `header.${encodedPayload}.signature`;
            window.sessionStorage.setItem("eden_token", token);
            const result = decodeToken();
            expect(result).toEqual(payload);
        });
    });
});
