import { describe, it, expect, beforeEach, afterEach, vi } from "vitest";
import { generateShortcut } from "../keys";
describe("Keys Service", () => {
    let originalNavigator;
    beforeEach(() => {
        originalNavigator = global.navigator;
    });
    afterEach(() => {
        global.navigator = originalNavigator;
    });
    describe("generateShortcut", () => {
        it("should generate shortcut with regular keys", () => {
            const result = generateShortcut("a", "b");
            expect(result).toContain("class='key'");
            expect(result).toContain(">a<");
            expect(result).toContain(">b<");
        });
        it("should replace backspace with symbol", () => {
            const result = generateShortcut("backspace");
            expect(result).toContain("⌫");
            expect(result).not.toContain("backspace");
        });
        it("should replace enter with symbol", () => {
            const result = generateShortcut("enter");
            expect(result).toContain("↵");
            expect(result).not.toContain("enter");
        });
        it("should replace shift with symbol", () => {
            const result = generateShortcut("shift");
            expect(result).toContain("⇧");
            expect(result).not.toContain("shift");
        });
        it("should replace ctrl with cmd symbol on Mac", () => {
            Object.defineProperty(global.navigator, "userAgent", {
                value: "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7)",
                writable: true,
                configurable: true,
            });
            vi.resetModules();
            const result = generateShortcut("ctrl");
            expect(result).toContain("class='key'");
        });
        it("should handle multiple keys", () => {
            const result = generateShortcut("ctrl", "shift", "a");
            expect(result).toContain("class='shortcut'");
            const keyMatches = result.match(/class='key'/g);
            expect(keyMatches).toHaveLength(3);
        });
        it("should wrap result in shortcut span", () => {
            const result = generateShortcut("a");
            expect(result).toContain("class='shortcut'");
        });
        it("should handle single key", () => {
            const result = generateShortcut("a");
            expect(result).toContain(">a<");
            expect(result).toContain("class='key'");
        });
        it("should handle special key combinations", () => {
            const result = generateShortcut("shift", "enter");
            expect(result).toContain("⇧");
            expect(result).toContain("↵");
        });
        it("should handle numeric keys", () => {
            const result = generateShortcut("1", "2", "3");
            expect(result).toContain(">1<");
            expect(result).toContain(">2<");
            expect(result).toContain(">3<");
        });
        it("should handle empty input", () => {
            const result = generateShortcut();
            expect(result).toContain("class='shortcut'");
        });
    });
});
